import numpy as np
import matplotlib.pyplot as plt
from mpl_toolkits.mplot3d.art3d import Poly3DCollection
from matplotlib.backends.backend_agg import FigureCanvasAgg as FigureCanvas

def make_small_stellated_dodecahedron(angle_x=0, angle_y=0, angle_z=0, img_size=400, alpha=0.9):
    angle_x = float(angle_x)
    angle_y = float(angle_y)
    angle_z = float(angle_z)

    # Коэффициент C0
    C0 = (np.sqrt(5) - 1) / 4.0  # ≈ 0.30901699
    verts = np.array([
        ( 0.0,  0.5, -C0), ( 0.0,  0.5,  C0),
        ( 0.0, -0.5, -C0), ( 0.0, -0.5,  C0),
        ( 0.5, -C0,  0.0), (-0.5, -C0,  0.0),
        ( 0.5,  C0,  0.0), (-0.5,  C0, 0.0),
        (-C0, 0.0,  0.5), (-C0, 0.0, -0.5),
        ( C0, 0.0,  0.5), ( C0, 0.0, -0.5),
    ], dtype=float)

    # Грани — каждая пентаграмма задана 5 вершинами (по индексу)
    faces = [
        (0, 2, 7, 11, 5),
        (0, 5, 1, 9, 8),
        (0, 8, 6, 7, 10),
        (1, 3, 6, 8, 4),
        (1, 4, 0, 10, 11),
        (1, 11, 7, 6, 9),
        (2, 0, 4, 9, 6),
        (2, 6, 3, 11, 10),
        (2, 10, 5, 4, 8),
        (3, 1, 5, 10, 7),
        (3, 7, 2, 8, 9),
        (3, 9, 4, 5, 11),
    ]

    # Rotation matrices
    ax = np.radians(angle_x)
    ay = np.radians(angle_y)
    az = np.radians(angle_z)
    Rx = np.array([[1, 0, 0], [0, np.cos(ax), -np.sin(ax)], [0, np.sin(ax), np.cos(ax)]])
    Ry = np.array([[np.cos(ay), 0, np.sin(ay)], [0,1,0], [-np.sin(ay),0, np.cos(ay)]])
    Rz = np.array([[np.cos(az), -np.sin(az), 0], [np.sin(az), np.cos(az), 0], [0,0,1]])
    R = Rz @ Ry @ Rx
    verts_rot = (R @ verts.T).T

    # Рисование
    dpi = 100
    fig = plt.figure(figsize=(img_size/dpi, img_size/dpi), dpi=dpi)
    fig.patch.set_alpha(0.0)  # transparent
    canvas = FigureCanvas(fig)
    ax3 = fig.add_subplot(111, projection='3d')
    ax3.set_facecolor((0, 0, 0, 0))

    poly_verts = [[verts_rot[i] for i in face] for face in faces]
    face_colors = [(0.2, 0.7, 0.2, alpha)] * len(poly_verts)  # одинаковый цвет — светло-зелёный

    collection = Poly3DCollection(poly_verts, linewidths=0.6, edgecolors='k', alpha=alpha)
    collection.set_facecolor(face_colors)
    ax3.add_collection3d(collection)

    max_range = np.max(np.abs(verts_rot))
    ax3.set_xlim(-max_range, max_range)
    ax3.set_ylim(-max_range, max_range)
    ax3.set_zlim(-max_range, max_range)
    ax3.set_box_aspect([1,1,1])
    ax3.set_axis_off()
    plt.subplots_adjust(0,0,1,1)
    fig.canvas.draw()

    h, w = fig.canvas.get_width_height()
    rgba = np.frombuffer(fig.canvas.buffer_rgba(), dtype=np.uint8).reshape((h, w, 4))
    bgra = rgba[:, :, [2,1,0,3]].copy()
    plt.close(fig)
    return bgra

def execute(params, inputs, outputs):
    outputs.m1 = make_small_stellated_dodecahedron(params.a, params.b, params.c)
    return "Hello! figure is in m1"

# Example
if __name__ == "__main__":
    img = make_small_stellated_dodecahedron(30, 30, 30)
    from PIL import Image
    Image.fromarray(img).show()
