/*
 * Decompiled with CFR 0.152.
 */
package net.algart.arrays;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Objects;
import net.algart.arrays.JArrays;
import net.algart.arrays.PackedBitArrays;
import net.algart.arrays.TooLargeArrayException;

public class PackedBitArraysPer8 {
    private static final byte[] REVERSE = new byte[]{0, -128, 64, -64, 32, -96, 96, -32, 16, -112, 80, -48, 48, -80, 112, -16, 8, -120, 72, -56, 40, -88, 104, -24, 24, -104, 88, -40, 56, -72, 120, -8, 4, -124, 68, -60, 36, -92, 100, -28, 20, -108, 84, -44, 52, -76, 116, -12, 12, -116, 76, -52, 44, -84, 108, -20, 28, -100, 92, -36, 60, -68, 124, -4, 2, -126, 66, -62, 34, -94, 98, -30, 18, -110, 82, -46, 50, -78, 114, -14, 10, -118, 74, -54, 42, -86, 106, -22, 26, -102, 90, -38, 58, -70, 122, -6, 6, -122, 70, -58, 38, -90, 102, -26, 22, -106, 86, -42, 54, -74, 118, -10, 14, -114, 78, -50, 46, -82, 110, -18, 30, -98, 94, -34, 62, -66, 126, -2, 1, -127, 65, -63, 33, -95, 97, -31, 17, -111, 81, -47, 49, -79, 113, -15, 9, -119, 73, -55, 41, -87, 105, -23, 25, -103, 89, -39, 57, -71, 121, -7, 5, -123, 69, -59, 37, -91, 101, -27, 21, -107, 85, -43, 53, -75, 117, -11, 13, -115, 77, -51, 45, -83, 109, -19, 29, -99, 93, -35, 61, -67, 125, -3, 3, -125, 67, -61, 35, -93, 99, -29, 19, -109, 83, -45, 51, -77, 115, -13, 11, -117, 75, -53, 43, -85, 107, -21, 27, -101, 91, -37, 59, -69, 123, -5, 7, -121, 71, -57, 39, -89, 103, -25, 23, -105, 87, -41, 55, -73, 119, -9, 15, -113, 79, -49, 47, -81, 111, -17, 31, -97, 95, -33, 63, -65, 127, -1};

    private PackedBitArraysPer8() {
    }

    public static long[] toLongArray(byte[] byteArray) {
        Objects.requireNonNull(byteArray, "Null byte[] array");
        return PackedBitArraysPer8.toLongArray(null, byteArray, 8L * (long)byteArray.length);
    }

    public static long[] toLongArray(byte[] byteArray, long numberOfBits) {
        return PackedBitArraysPer8.toLongArray(null, byteArray, numberOfBits);
    }

    public static long[] toLongArray(long[] result, byte[] byteArray, long numberOfBits) {
        int cntFinish;
        Objects.requireNonNull(byteArray, "Null byte[] array");
        if (numberOfBits < 0L) {
            throw new IllegalArgumentException("Negative numberOfBits = " + numberOfBits);
        }
        long numberOfBytes = PackedBitArraysPer8.packedLength(numberOfBits);
        if (numberOfBytes > (long)byteArray.length) {
            throw new IllegalArgumentException("Too short byte[" + byteArray.length + "] array (packed bits): it must contain at least " + numberOfBytes + " byte elements to contain " + numberOfBits + " bits");
        }
        int numberOfLongs = (int)PackedBitArrays.packedLength(numberOfBits);
        if (result == null) {
            result = new long[numberOfLongs];
        } else if (numberOfLongs > result.length) {
            throw new IllegalArgumentException("Too short result array long[" + result.length + "]: it must contain at least " + numberOfLongs + " elements to store " + numberOfBits + " bits");
        }
        int numberOfWholeLongs = (int)(numberOfBytes >>> 3);
        int alignedLength = numberOfWholeLongs << 3;
        ByteBuffer bb = ByteBuffer.wrap(byteArray);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        bb.limit(alignedLength);
        bb.asLongBuffer().get(result, 0, numberOfWholeLongs);
        if (numberOfBytes > (long)alignedLength) {
            assert (numberOfLongs == numberOfWholeLongs + 1);
            long v = 0L;
            int k = alignedLength;
            int shift = 0;
            while ((long)k < numberOfBytes) {
                v |= ((long)byteArray[k] & 0xFFL) << shift;
                ++k;
                shift += 8;
            }
            result[numberOfWholeLongs] = v;
        }
        if ((cntFinish = (int)(numberOfBits & 0x3FL)) > 0) {
            int n = numberOfLongs - 1;
            result[n] = result[n] & (1L << cntFinish) - 1L;
        }
        return result;
    }

    public static long[] toLongArray(ByteBuffer byteBuffer) {
        Objects.requireNonNull(byteBuffer, "Null ByteBuffer");
        return PackedBitArraysPer8.toLongArray(null, byteBuffer, 8L * (long)byteBuffer.limit());
    }

    public static long[] toLongArray(ByteBuffer byteBuffer, long numberOfBits) {
        return PackedBitArraysPer8.toLongArray(null, byteBuffer, numberOfBits);
    }

    public static long[] toLongArray(long[] result, ByteBuffer byteBuffer, long numberOfBits) {
        int cntFinish;
        Objects.requireNonNull(byteBuffer, "Null ByteBuffer");
        if (numberOfBits < 0L) {
            throw new IllegalArgumentException("Negative numberOfBits = " + numberOfBits);
        }
        ByteBuffer bb = byteBuffer.duplicate();
        long numberOfBytes = PackedBitArraysPer8.packedLength(numberOfBits);
        if (numberOfBytes > (long)bb.limit()) {
            throw new IllegalArgumentException("Too short ByteBuffer, " + bb.limit() + " bytes (packed bits): it must contain at least " + numberOfBytes + " byte elements to contain " + numberOfBits + " bits");
        }
        int numberOfLongs = (int)PackedBitArrays.packedLength(numberOfBits);
        if (result == null) {
            result = new long[numberOfLongs];
        } else if (numberOfLongs > result.length) {
            throw new IllegalArgumentException("Too short result array long[" + result.length + "]: it must contain at least " + numberOfLongs + " elements to store " + numberOfBits + " bits");
        }
        int numberOfWholeLongs = (int)(numberOfBytes >>> 3);
        int alignedLength = numberOfWholeLongs << 3;
        bb.order(ByteOrder.LITTLE_ENDIAN);
        bb.rewind();
        bb.limit(alignedLength);
        bb.asLongBuffer().get(result, 0, numberOfWholeLongs);
        bb.limit((int)numberOfBytes);
        if (numberOfBytes > (long)alignedLength) {
            assert (numberOfLongs == numberOfWholeLongs + 1);
            long v = 0L;
            int k = alignedLength;
            int shift = 0;
            while ((long)k < numberOfBytes) {
                v |= (long)(bb.get(k) & 0xFF) << shift;
                ++k;
                shift += 8;
            }
            result[numberOfWholeLongs] = v;
        }
        if ((cntFinish = (int)(numberOfBits & 0x3FL)) > 0) {
            int n = numberOfLongs - 1;
            result[n] = result[n] & (1L << cntFinish) - 1L;
        }
        return result;
    }

    public static byte[] toByteArray(long[] longArray, long numberOfBits) {
        return PackedBitArraysPer8.toByteArray(null, longArray, numberOfBits);
    }

    public static byte[] toByteArray(byte[] result, long[] longArray, long numberOfBits) {
        int cntFinish;
        Objects.requireNonNull(longArray, "Null long[] array (packed bits)");
        if (numberOfBits < 0L) {
            throw new IllegalArgumentException("Negative numberOfBits = " + numberOfBits);
        }
        long numberOfLongs = PackedBitArrays.packedLength(numberOfBits);
        if (numberOfLongs > (long)longArray.length) {
            throw new IllegalArgumentException("Too short long[" + longArray.length + "] array (packed bits): it must contain at least " + numberOfLongs + " long elements to contain " + numberOfBits + " bits");
        }
        long numberOfBytes = PackedBitArraysPer8.packedLength(numberOfBits);
        if (numberOfBytes > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large number of bits " + numberOfBits + " > 2^34-1: cannot pack such a large bit array into byte[] array");
        }
        if (result == null) {
            result = new byte[(int)numberOfBytes];
        } else if (numberOfBytes > (long)result.length) {
            throw new IllegalArgumentException("Too short result array byte[" + result.length + "]: it must contain at least " + numberOfBytes + " elements to store " + numberOfBits + " bits");
        }
        int numberOfWholeLongs = (int)(numberOfBytes >>> 3);
        int alignedLength = numberOfWholeLongs << 3;
        ByteBuffer bb = ByteBuffer.wrap(result);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        bb.asLongBuffer().put(longArray, 0, numberOfWholeLongs);
        if (numberOfBytes > (long)alignedLength) {
            assert (numberOfLongs == (long)(numberOfWholeLongs + 1));
            long v = longArray[numberOfWholeLongs];
            int k = alignedLength;
            int shift = 0;
            while ((long)k < numberOfBytes) {
                result[k] = (byte)(v >>> shift);
                ++k;
                shift += 8;
            }
        }
        if ((cntFinish = (int)(numberOfBits & 7L)) > 0) {
            int n = (int)numberOfBytes - 1;
            result[n] = (byte)(result[n] & (byte)((1 << cntFinish) - 1));
        }
        return result;
    }

    public static long unpackedLength(long numberOfBytes) {
        if (numberOfBytes < 0L) {
            throw new IllegalArgumentException("Negative packed length");
        }
        if (numberOfBytes >= 0x1000000000000000L) {
            throw new TooLargeArrayException("Too large packed length: number of unpacked bits >= 2^60");
        }
        return numberOfBytes << 3;
    }

    public static long unpackedLength(byte[] array) {
        return (long)array.length << 3;
    }

    public static long packedLength(long numberOfBits) {
        if (numberOfBits < 0L) {
            throw new IllegalArgumentException("Negative unpacked length");
        }
        return numberOfBits + 7L >>> 3;
    }

    public static int packedLength32(long numberOfBits) {
        if (numberOfBits < 0L) {
            throw new IllegalArgumentException("Negative unpacked length");
        }
        long result = numberOfBits + 7L >>> 3;
        if (result > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large number of bits " + numberOfBits + ": number of packed longs " + result + " >= 2^31");
        }
        return (int)result;
    }

    public static int packedLength32(int numberOfBits) {
        if (numberOfBits < 0) {
            throw new IllegalArgumentException("Negative unpacked length");
        }
        return numberOfBits + 7 >>> 3;
    }

    public static boolean getBit(byte[] src, long index) {
        return (src[(int)(index >>> 3)] & 1 << ((int)index & 7)) != 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void setBit(byte[] dest, long index, boolean value) {
        byte[] byArray = dest;
        synchronized (dest) {
            if (value) {
                int n = (int)(index >>> 3);
                dest[n] = (byte)(dest[n] | (byte)(1 << ((int)index & 7)));
            } else {
                int n = (int)(index >>> 3);
                dest[n] = (byte)(dest[n] & (byte)(~(1 << ((int)index & 7))));
            }
            // ** MonitorExit[var4_3] (shouldn't be in output)
            return;
        }
    }

    public static void setBitNoSync(byte[] dest, long index, boolean value) {
        if (value) {
            int n = (int)(index >>> 3);
            dest[n] = (byte)(dest[n] | (byte)(1 << ((int)index & 7)));
        } else {
            int n = (int)(index >>> 3);
            dest[n] = (byte)(dest[n] & (byte)(~(1 << ((int)index & 7))));
        }
    }

    public static long getBits64(byte[] src, long srcPos, int count) {
        long result;
        block6: {
            long actualBits;
            int bitsLeft;
            Objects.requireNonNull(src, "Null src");
            if (srcPos < 0L) {
                throw new IndexOutOfBoundsException("Negative srcPos argument: " + srcPos);
            }
            if (count < 0) {
                throw new IllegalArgumentException("Negative count argument: " + count);
            }
            if (count > 64) {
                throw new IllegalArgumentException("Too large count argument: " + count + "; we cannot get > 64 bits in getBits64 method");
            }
            long srcPosDiv8 = srcPos >>> 3;
            if (count == 0 || srcPosDiv8 >= (long)src.length) {
                return 0L;
            }
            result = 0L;
            int sPosRem = (int)(srcPos & 7L);
            int sPos = (int)srcPosDiv8;
            int shift = 0;
            while (count > (bitsLeft = 8 - sPosRem)) {
                actualBits = ((long)src[sPos] & 0xFFL) >>> sPosRem;
                result |= actualBits << shift;
                count -= bitsLeft;
                shift += bitsLeft;
                if (++sPos < src.length) {
                    sPosRem = 0;
                    continue;
                }
                break block6;
            }
            actualBits = ((long)src[sPos] & 255L >>> bitsLeft - count) >>> sPosRem;
            result |= actualBits << shift;
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void setBits64(byte[] dest, long destPos, long bits, int count) {
        Objects.requireNonNull(dest, "Null dest");
        if (destPos < 0L) {
            throw new IndexOutOfBoundsException("Negative destPos argument: " + destPos);
        }
        if (count < 0) {
            throw new IllegalArgumentException("Negative count argument: " + count);
        }
        if (count > 64) {
            throw new IllegalArgumentException("Too large count argument: " + count + "; we cannot set > 64 bits in setBits64 method");
        }
        long destPosDiv8 = destPos >>> 3;
        if (count == 0 || destPosDiv8 >= (long)dest.length) {
            return;
        }
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if (cntStart > count) {
            cntStart = count;
            maskStart &= (1 << dPosRem + count) - 1;
        }
        int dPos = (int)destPosDiv8;
        byte[] byArray = dest;
        synchronized (dest) {
            if (cntStart > 0) {
                dest[dPos] = (byte)(bits << dPosRem & (long)maskStart | (long)(dest[dPos] & ~maskStart));
                ++dPos;
                count -= cntStart;
                bits >>>= cntStart;
            }
            while (count >= 8) {
                if (dPos >= dest.length) {
                    // ** MonitorExit[var12_9] (shouldn't be in output)
                    return;
                }
                dest[dPos++] = (byte)bits;
                count -= 8;
                bits >>>= 8;
            }
            if (count > 0 && dPos < dest.length) {
                int maskFinish = (1 << count) - 1;
                dest[dPos] = (byte)(bits & (long)maskFinish | (long)(dest[dPos] & ~maskFinish));
            }
            // ** MonitorExit[var12_9] (shouldn't be in output)
            return;
        }
    }

    public static void setBits64NoSync(byte[] dest, long destPos, long bits, int count) {
        Objects.requireNonNull(dest, "Null dest");
        if (destPos < 0L) {
            throw new IndexOutOfBoundsException("Negative destPos argument: " + destPos);
        }
        if (count < 0) {
            throw new IllegalArgumentException("Negative count argument: " + count);
        }
        if (count > 64) {
            throw new IllegalArgumentException("Too large count argument: " + count + "; we cannot set > 64 bits in setBits64NoSync method");
        }
        long destPosDiv8 = destPos >>> 3;
        if (count == 0 || destPosDiv8 >= (long)dest.length) {
            return;
        }
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if (cntStart > count) {
            cntStart = count;
            maskStart &= (1 << dPosRem + count) - 1;
        }
        int dPos = (int)destPosDiv8;
        if (cntStart > 0) {
            dest[dPos] = (byte)(bits << dPosRem & (long)maskStart | (long)(dest[dPos] & ~maskStart));
            ++dPos;
            count -= cntStart;
            bits >>>= cntStart;
        }
        while (count >= 8) {
            if (dPos >= dest.length) {
                return;
            }
            dest[dPos++] = (byte)bits;
            count -= 8;
            bits >>>= 8;
        }
        if (count > 0 && dPos < dest.length) {
            int maskFinish = (1 << count) - 1;
            dest[dPos] = (byte)(bits & (long)maskFinish | (long)(dest[dPos] & ~maskFinish));
        }
    }

    public static boolean getBitInReverseOrder(byte[] src, long index) {
        return (src[(int)(index >>> 3)] & 1 << 7 - ((int)index & 7)) != 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void setBitInReverseOrder(byte[] dest, long index, boolean value) {
        byte[] byArray = dest;
        synchronized (dest) {
            int bitIndex = 7 - ((int)index & 7);
            if (value) {
                int n = (int)(index >>> 3);
                dest[n] = (byte)(dest[n] | (byte)(1 << bitIndex));
            } else {
                int n = (int)(index >>> 3);
                dest[n] = (byte)(dest[n] & (byte)(~(1 << bitIndex)));
            }
            // ** MonitorExit[var4_3] (shouldn't be in output)
            return;
        }
    }

    public static void setBitInReverseOrderNoSync(byte[] dest, long index, boolean value) {
        int bitIndex = 7 - ((int)index & 7);
        if (value) {
            int n = (int)(index >>> 3);
            dest[n] = (byte)(dest[n] | (byte)(1 << bitIndex));
        } else {
            int n = (int)(index >>> 3);
            dest[n] = (byte)(dest[n] & (byte)(~(1 << bitIndex)));
        }
    }

    public static long getBits64InReverseOrder(byte[] src, long srcPos, int count) {
        long actualBits;
        int bitsLeft;
        int sPos;
        int sPosRem;
        long result;
        block5: {
            Objects.requireNonNull(src, "Null src");
            if (srcPos < 0L) {
                throw new IndexOutOfBoundsException("Negative srcPos argument: " + srcPos);
            }
            if (count < 0) {
                throw new IllegalArgumentException("Negative count argument: " + count);
            }
            if (count > 64) {
                throw new IllegalArgumentException("Too large count argument: " + count + "; we cannot get > 64 bits in getBits64 method");
            }
            long srcPosDiv8 = srcPos >>> 3;
            if (count == 0 || srcPosDiv8 >= (long)src.length) {
                return 0L;
            }
            result = 0L;
            sPosRem = (int)(srcPos & 7L);
            sPos = (int)srcPosDiv8;
            do {
                if (count <= (bitsLeft = 8 - sPosRem)) break block5;
                actualBits = (long)src[sPos] & 255L >>> sPosRem;
                result = result << bitsLeft | actualBits;
                count -= bitsLeft;
                sPosRem = 0;
            } while (++sPos < src.length);
            return result << count;
        }
        actualBits = (long)src[sPos] & (long)(~(65280 >>> sPosRem)) & 0xFFL;
        result = result << count | actualBits >> bitsLeft - count;
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void setBits64InReverseOrder(byte[] dest, long destPos, long bits, int count) {
        Objects.requireNonNull(dest, "Null dest");
        if (destPos < 0L) {
            throw new IndexOutOfBoundsException("Negative destPos argument: " + destPos);
        }
        if (count < 0) {
            throw new IllegalArgumentException("Negative count argument: " + count);
        }
        if (count > 64) {
            throw new IllegalArgumentException("Too large count argument: " + count + "; we cannot set > 64 bits in setBits64InReverseOrder method");
        }
        long destPosDiv8 = destPos >>> 3;
        if (count == 0 || destPosDiv8 >= (long)dest.length) {
            return;
        }
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if (cntStart > count) {
            cntStart = count;
            maskStart &= 65280 >>> dPosRem + count;
        }
        int dPos = (int)destPosDiv8;
        byte[] byArray = dest;
        synchronized (dest) {
            if (cntStart > 0) {
                int shift = count - (8 - dPosRem);
                long v = shift >= 0 ? bits >>> shift : bits << -shift;
                dest[dPos] = (byte)(v & (long)maskStart | (long)(dest[dPos] & ~maskStart));
                ++dPos;
                count -= cntStart;
            }
            while (count >= 8) {
                if (dPos >= dest.length) {
                    // ** MonitorExit[var12_9] (shouldn't be in output)
                    return;
                }
                dest[dPos++] = (byte)(bits >>> (count -= 8));
            }
            if (count > 0 && dPos < dest.length) {
                int maskFinish = 65280 >>> count;
                dest[dPos] = (byte)(bits << 8 - count & (long)maskFinish | (long)(dest[dPos] & ~maskFinish));
            }
            // ** MonitorExit[var12_9] (shouldn't be in output)
            return;
        }
    }

    public static void setBits64InReverseOrderNoSync(byte[] dest, long destPos, long bits, int count) {
        Objects.requireNonNull(dest, "Null dest");
        if (destPos < 0L) {
            throw new IndexOutOfBoundsException("Negative destPos argument: " + destPos);
        }
        if (count < 0) {
            throw new IllegalArgumentException("Negative count argument: " + count);
        }
        if (count > 64) {
            throw new IllegalArgumentException("Too large count argument: " + count + "; we cannot set > 64 bits in setBits64InReverseOrderNoSync method");
        }
        long destPosDiv8 = destPos >>> 3;
        if (count == 0 || destPosDiv8 >= (long)dest.length) {
            return;
        }
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if (cntStart > count) {
            cntStart = count;
            maskStart &= 65280 >>> dPosRem + cntStart;
        }
        int dPos = (int)destPosDiv8;
        if (cntStart > 0) {
            int shift = count - (8 - dPosRem);
            long v = shift >= 0 ? bits >>> shift : bits << -shift;
            dest[dPos] = (byte)(v & (long)maskStart | (long)(dest[dPos] & ~maskStart));
            ++dPos;
            count -= cntStart;
        }
        while (count >= 8) {
            if (dPos >= dest.length) {
                return;
            }
            dest[dPos++] = (byte)(bits >>> (count -= 8));
        }
        if (count > 0 && dPos < dest.length) {
            int maskFinish = 65280 >>> count;
            dest[dPos] = (byte)(bits << 8 - count & (long)maskFinish | (long)(dest[dPos] & ~maskFinish));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void copyBits(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (src == dest && srcPos <= destPos && srcPos + count > destPos) {
            int sPrev;
            if (sPosRem == dPosRem) {
                if (sPos == dPos) {
                    return;
                }
                int sPosStart = sPos++;
                int dPosStart = dPos++;
                if (cntStart > 0) {
                    count -= (long)cntStart;
                }
                int cnt = (int)(count >>> 3);
                int cntFinish = (int)(count & 7L);
                if (cntFinish > 0) {
                    int maskFinish = (1 << cntFinish) - 1;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos + cnt] = (byte)(src[sPos + cnt] & maskFinish | dest[dPos + cnt] & ~maskFinish);
                    // MONITOREXIT : byArray
                }
                System.arraycopy(src, sPos, dest, dPos, cnt);
                if (cntStart <= 0) return;
                byte[] maskFinish = dest;
                // MONITORENTER : dest
                dest[dPosStart] = (byte)(src[sPosStart] & maskStart | dest[dPosStart] & ~maskStart);
                // MONITOREXIT : maskFinish
                return;
            }
            int shift = dPosRem - sPosRem;
            int sPosStart = sPos++;
            int dPosStart = dPos;
            int sPosRemStart = sPosRem;
            if (cntStart > 0) {
                sPosRem = sPosRem + cntStart <= 8 ? (sPosRem += cntStart) : sPosRem + cntStart & 7;
                count -= (long)cntStart;
                ++dPos;
            }
            int cnt = (int)(count >>> 3);
            long dPosMin = dPos;
            sPos += cnt;
            dPos += cnt;
            int cntFinish = (int)(count & 7L);
            int sPosRem8 = 8 - sPosRem;
            if (cntFinish > 0) {
                int v;
                int maskFinish = (1 << cntFinish) - 1;
                if (sPosRem + cntFinish <= 8) {
                    sPrev = src[sPos] & 0xFF;
                    v = sPrev >>> sPosRem;
                } else {
                    sPrev = src[sPos] & 0xFF;
                    v = sPrev >>> sPosRem | (src[sPos + 1] & 0xFF) << sPosRem8;
                }
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
                // MONITOREXIT : byArray
            } else {
                sPrev = src[sPos] & 0xFF;
            }
            while ((long)dPos > dPosMin) {
                int n = sPrev << sPosRem8;
                sPrev = src[--sPos] & 0xFF;
                dest[--dPos] = (byte)(n | sPrev >>> sPosRem);
            }
            if (cntStart <= 0) return;
            int v = sPosRemStart + cntStart <= 8 ? (shift > 0 ? (src[sPosStart] & 0xFF) << shift : (src[sPosStart] & 0xFF) >>> -shift) : (src[sPosStart] & 0xFF) >>> -shift | (src[sPosStart + 1] & 0xFF) << 8 + shift;
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPosStart] = (byte)(v & maskStart | dest[dPosStart] & ~maskStart);
            // MONITOREXIT : byArray
            return;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] shift = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)(src[sPos] & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : shift
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int cnt = (int)(count >>> 3);
            System.arraycopy(src, sPos, dest, dPos, cnt);
            sPos += cnt;
            dPos += cnt;
            int cntFinish = (int)(count & 7L);
            if (cntFinish <= 0) return;
            int maskFinish = (1 << cntFinish) - 1;
            byte[] sPosRemStart = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)(src[sPos] & maskFinish | dest[dPos] & ~maskFinish);
            // MONITOREXIT : sPosRemStart
            return;
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) >>> -shift | sNext << 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] sPosRemStart = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)(v & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : sPosRemStart
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = sNext >>> sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[dPos] = (byte)(n | sNext << sPosRem8);
            ++dPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = (1 << cntFinish) - 1;
        int v = sPosRem + cntFinish <= 8 ? sNext >>> sPosRem : sNext >>> sPosRem | (src[sPos + 1] & 0xFF) << sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    public static void copyBitsNoSync(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (src == dest && srcPos <= destPos && srcPos + count > destPos) {
            if (sPosRem == dPosRem) {
                if (sPos == dPos) {
                    return;
                }
                int sPosStart = sPos++;
                int dPosStart = dPos++;
                if (cntStart > 0) {
                    count -= (long)cntStart;
                }
                int cnt = (int)(count >>> 3);
                int cntFinish = (int)(count & 7L);
                if (cntFinish > 0) {
                    int maskFinish = (1 << cntFinish) - 1;
                    dest[dPos + cnt] = (byte)(src[sPos + cnt] & maskFinish | dest[dPos + cnt] & ~maskFinish);
                }
                System.arraycopy(src, sPos, dest, dPos, cnt);
                if (cntStart > 0) {
                    dest[dPosStart] = (byte)(src[sPosStart] & maskStart | dest[dPosStart] & ~maskStart);
                }
            } else {
                int sPrev;
                int shift = dPosRem - sPosRem;
                int sPosStart = sPos++;
                int dPosStart = dPos;
                int sPosRemStart = sPosRem;
                if (cntStart > 0) {
                    sPosRem = sPosRem + cntStart <= 8 ? (sPosRem += cntStart) : sPosRem + cntStart & 7;
                    count -= (long)cntStart;
                    ++dPos;
                }
                int cnt = (int)(count >>> 3);
                long dPosMin = dPos;
                sPos += cnt;
                dPos += cnt;
                int cntFinish = (int)(count & 7L);
                int sPosRem8 = 8 - sPosRem;
                if (cntFinish > 0) {
                    int v;
                    int maskFinish = (1 << cntFinish) - 1;
                    if (sPosRem + cntFinish <= 8) {
                        sPrev = src[sPos] & 0xFF;
                        v = sPrev >>> sPosRem;
                    } else {
                        sPrev = src[sPos] & 0xFF;
                        v = sPrev >>> sPosRem | (src[sPos + 1] & 0xFF) << sPosRem8;
                    }
                    dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
                } else {
                    sPrev = src[sPos] & 0xFF;
                }
                while ((long)dPos > dPosMin) {
                    int n = sPrev << sPosRem8;
                    sPrev = src[--sPos] & 0xFF;
                    dest[--dPos] = (byte)(n | sPrev >>> sPosRem);
                }
                if (cntStart > 0) {
                    int v = sPosRemStart + cntStart <= 8 ? (shift > 0 ? (src[sPosStart] & 0xFF) << shift : (src[sPosStart] & 0xFF) >>> -shift) : (src[sPosStart] & 0xFF) >>> -shift | (src[sPosStart + 1] & 0xFF) << 8 + shift;
                    dest[dPosStart] = (byte)(v & maskStart | dest[dPosStart] & ~maskStart);
                }
            }
        } else if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                dest[dPos] = (byte)(src[sPos] & 0xFF & maskStart | dest[dPos] & ~maskStart);
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int cnt = (int)(count >>> 3);
            System.arraycopy(src, sPos, dest, dPos, cnt);
            sPos += cnt;
            dPos += cnt;
            int cntFinish = (int)(count & 7L);
            if (cntFinish > 0) {
                int maskFinish = (1 << cntFinish) - 1;
                dest[dPos] = (byte)(src[sPos] & 0xFF & maskFinish | dest[dPos] & ~maskFinish);
            }
        } else {
            int sNext;
            int shift = dPosRem - sPosRem;
            if (cntStart > 0) {
                int v;
                if (sPosRem + cntStart <= 8) {
                    if (shift > 0) {
                        sNext = src[sPos] & 0xFF;
                        v = sNext << shift;
                    } else {
                        sNext = src[sPos] & 0xFF;
                        v = sNext >>> -shift;
                    }
                    sPosRem += cntStart;
                } else {
                    sNext = src[sPos + 1] & 0xFF;
                    v = (src[sPos] & 0xFF) >>> -shift | sNext << 8 + shift;
                    ++sPos;
                    sPosRem = sPosRem + cntStart & 7;
                }
                dest[dPos] = (byte)(v & maskStart | dest[dPos] & ~maskStart);
                if ((count -= (long)cntStart) == 0L) {
                    return;
                }
                ++dPos;
            } else {
                if (count == 0L) {
                    return;
                }
                sNext = src[sPos] & 0xFF;
            }
            int sPosRem8 = 8 - sPosRem;
            int dPosMax = dPos + (int)(count >>> 3);
            while (dPos < dPosMax) {
                int n = sNext >>> sPosRem;
                sNext = src[++sPos] & 0xFF;
                dest[dPos] = (byte)(n | sNext << sPosRem8);
                ++dPos;
            }
            int cntFinish = (int)(count & 7L);
            if (cntFinish > 0) {
                int maskFinish = (1 << cntFinish) - 1;
                int v = sPosRem + cntFinish <= 8 ? sNext >>> sPosRem : sNext >>> sPosRem | (src[sPos + 1] & 0xFF) << sPosRem8;
                dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void copyBitsInReverseOrder(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> dPosRem + cntStart;
        }
        if (src == dest && srcPos <= destPos && srcPos + count > destPos) {
            int sPrev;
            if (sPosRem == dPosRem) {
                if (sPos == dPos) {
                    return;
                }
                int sPosStart = sPos++;
                int dPosStart = dPos++;
                if (cntStart > 0) {
                    count -= (long)cntStart;
                }
                int cnt = (int)(count >>> 3);
                int cntFinish = (int)(count & 7L);
                if (cntFinish > 0) {
                    int maskFinish = 65280 >>> cntFinish;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos + cnt] = (byte)(src[sPos + cnt] & maskFinish | dest[dPos + cnt] & ~maskFinish);
                    // MONITOREXIT : byArray
                }
                System.arraycopy(src, sPos, dest, dPos, cnt);
                if (cntStart <= 0) return;
                byte[] maskFinish = dest;
                // MONITORENTER : dest
                dest[dPosStart] = (byte)(src[sPosStart] & maskStart | dest[dPosStart] & ~maskStart);
                // MONITOREXIT : maskFinish
                return;
            }
            int shift = dPosRem - sPosRem;
            int sPosStart = sPos++;
            int dPosStart = dPos;
            int sPosRemStart = sPosRem;
            if (cntStart > 0) {
                sPosRem = sPosRem + cntStart <= 8 ? (sPosRem += cntStart) : sPosRem + cntStart & 7;
                count -= (long)cntStart;
                ++dPos;
            }
            int cnt = (int)(count >>> 3);
            long dPosMin = dPos;
            sPos += cnt;
            dPos += cnt;
            int cntFinish = (int)(count & 7L);
            int sPosRem8 = 8 - sPosRem;
            if (cntFinish > 0) {
                int v;
                int maskFinish = 65280 >>> cntFinish;
                if (sPosRem + cntFinish <= 8) {
                    sPrev = src[sPos] & 0xFF;
                    v = sPrev << sPosRem;
                } else {
                    sPrev = src[sPos] & 0xFF;
                    v = sPrev << sPosRem | (src[sPos + 1] & 0xFF) >>> sPosRem8;
                }
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
                // MONITOREXIT : byArray
            } else {
                sPrev = src[sPos] & 0xFF;
            }
            while ((long)dPos > dPosMin) {
                int n = sPrev >>> sPosRem8;
                sPrev = src[--sPos] & 0xFF;
                dest[--dPos] = (byte)(n | sPrev << sPosRem);
            }
            if (cntStart <= 0) return;
            int v = sPosRemStart + cntStart <= 8 ? (shift > 0 ? (src[sPosStart] & 0xFF) >>> shift : (src[sPosStart] & 0xFF) << -shift) : (src[sPosStart] & 0xFF) << -shift | (src[sPosStart + 1] & 0xFF) >>> 8 + shift;
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPosStart] = (byte)(v & maskStart | dest[dPosStart] & ~maskStart);
            // MONITOREXIT : byArray
            return;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] shift = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)(src[sPos] & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : shift
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int cnt = (int)(count >>> 3);
            System.arraycopy(src, sPos, dest, dPos, cnt);
            sPos += cnt;
            dPos += cnt;
            int cntFinish = (int)(count & 7L);
            if (cntFinish <= 0) return;
            int maskFinish = 65280 >>> cntFinish;
            byte[] sPosRemStart = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)(src[sPos] & maskFinish | dest[dPos] & ~maskFinish);
            // MONITOREXIT : sPosRemStart
            return;
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) << -shift | sNext >>> 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] sPosRemStart = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)(v & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : sPosRemStart
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = sNext << sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[dPos] = (byte)(n | sNext >>> sPosRem8);
            ++dPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = 65280 >>> cntFinish;
        int v = sPosRem + cntFinish <= 8 ? sNext << sPosRem : sNext << sPosRem | (src[sPos + 1] & 0xFF) >>> sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    public static void copyBitsInReverseOrderNoSync(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> dPosRem + cntStart;
        }
        if (src == dest && srcPos <= destPos && srcPos + count > destPos) {
            if (sPosRem == dPosRem) {
                if (sPos == dPos) {
                    return;
                }
                int sPosStart = sPos++;
                int dPosStart = dPos++;
                if (cntStart > 0) {
                    count -= (long)cntStart;
                }
                int cnt = (int)(count >>> 3);
                int cntFinish = (int)(count & 7L);
                if (cntFinish > 0) {
                    int maskFinish = 65280 >>> cntFinish;
                    dest[dPos + cnt] = (byte)(src[sPos + cnt] & maskFinish | dest[dPos + cnt] & ~maskFinish);
                }
                System.arraycopy(src, sPos, dest, dPos, cnt);
                if (cntStart > 0) {
                    dest[dPosStart] = (byte)(src[sPosStart] & maskStart | dest[dPosStart] & ~maskStart);
                }
            } else {
                int sPrev;
                int shift = dPosRem - sPosRem;
                int sPosStart = sPos++;
                int dPosStart = dPos;
                int sPosRemStart = sPosRem;
                if (cntStart > 0) {
                    sPosRem = sPosRem + cntStart <= 8 ? (sPosRem += cntStart) : sPosRem + cntStart & 7;
                    count -= (long)cntStart;
                    ++dPos;
                }
                int cnt = (int)(count >>> 3);
                long dPosMin = dPos;
                sPos += cnt;
                dPos += cnt;
                int cntFinish = (int)(count & 7L);
                int sPosRem8 = 8 - sPosRem;
                if (cntFinish > 0) {
                    int v;
                    int maskFinish = 65280 >>> cntFinish;
                    if (sPosRem + cntFinish <= 8) {
                        sPrev = src[sPos] & 0xFF;
                        v = sPrev << sPosRem;
                    } else {
                        sPrev = src[sPos] & 0xFF;
                        v = sPrev << sPosRem | (src[sPos + 1] & 0xFF) >>> sPosRem8;
                    }
                    dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
                } else {
                    sPrev = src[sPos] & 0xFF;
                }
                while ((long)dPos > dPosMin) {
                    int n = sPrev >>> sPosRem8;
                    sPrev = src[--sPos] & 0xFF;
                    dest[--dPos] = (byte)(n | sPrev << sPosRem);
                }
                if (cntStart > 0) {
                    int v = sPosRemStart + cntStart <= 8 ? (shift > 0 ? (src[sPosStart] & 0xFF) >>> shift : (src[sPosStart] & 0xFF) << -shift) : (src[sPosStart] & 0xFF) << -shift | (src[sPosStart + 1] & 0xFF) >>> 8 + shift;
                    dest[dPosStart] = (byte)(v & maskStart | dest[dPosStart] & ~maskStart);
                }
            }
        } else if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                dest[dPos] = (byte)(src[sPos] & maskStart | dest[dPos] & ~maskStart);
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int cnt = (int)(count >>> 3);
            System.arraycopy(src, sPos, dest, dPos, cnt);
            sPos += cnt;
            dPos += cnt;
            int cntFinish = (int)(count & 7L);
            if (cntFinish > 0) {
                int maskFinish = 65280 >>> cntFinish;
                dest[dPos] = (byte)(src[sPos] & maskFinish | dest[dPos] & ~maskFinish);
            }
        } else {
            int sNext;
            int shift = dPosRem - sPosRem;
            if (cntStart > 0) {
                int v;
                if (sPosRem + cntStart <= 8) {
                    if (shift > 0) {
                        sNext = src[sPos] & 0xFF;
                        v = sNext >>> shift;
                    } else {
                        sNext = src[sPos] & 0xFF;
                        v = sNext << -shift;
                    }
                    sPosRem += cntStart;
                } else {
                    sNext = src[sPos + 1] & 0xFF;
                    v = (src[sPos] & 0xFF) << -shift | sNext >>> 8 + shift;
                    ++sPos;
                    sPosRem = sPosRem + cntStart & 7;
                }
                dest[dPos] = (byte)(v & maskStart | dest[dPos] & ~maskStart);
                if ((count -= (long)cntStart) == 0L) {
                    return;
                }
                ++dPos;
            } else {
                if (count == 0L) {
                    return;
                }
                sNext = src[sPos] & 0xFF;
            }
            int sPosRem8 = 8 - sPosRem;
            int dPosMax = dPos + (int)(count >>> 3);
            while (dPos < dPosMax) {
                int n = sNext << sPosRem;
                sNext = src[++sPos] & 0xFF;
                dest[dPos] = (byte)(n | sNext >>> sPosRem8);
                ++dPos;
            }
            int cntFinish = (int)(count & 7L);
            if (cntFinish > 0) {
                int maskFinish = 65280 >>> cntFinish;
                int v = sPosRem + cntFinish <= 8 ? sNext << sPosRem : sNext << sPosRem | (src[sPos + 1] & 0xFF) >>> sPosRem8;
                dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void copyBitsFromReverseToNormalOrder(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)(REVERSE[src[sPos] & 0xFF] & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = (1 << cntFinish) - 1;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos] = (byte)(REVERSE[src[sPos] & 0xFF] & maskFinish | dest[dPos] & ~maskFinish);
                    // MONITOREXIT : byArray
                    return;
                }
                dest[dPos] = REVERSE[src[sPos] & 0xFF];
                ++dPos;
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
                    v = sNext << shift;
                } else {
                    sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
                    v = sNext >>> -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = REVERSE[src[sPos + 1] & 0xFF] & 0xFF;
                v = (REVERSE[src[sPos] & 0xFF] & 0xFF) >>> -shift | sNext << 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)(v & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = sNext >>> sPosRem;
            sNext = REVERSE[src[++sPos] & 0xFF] & 0xFF;
            dest[dPos] = (byte)(n | sNext << sPosRem8);
            ++dPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = (1 << cntFinish) - 1;
        int v = sPosRem + cntFinish <= 8 ? sNext >>> sPosRem : sNext >>> sPosRem | (REVERSE[src[sPos + 1] & 0xFF] & 0xFF) << sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    public static void copyBitsFromReverseToNormalOrderNoSync(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                dest[dPos] = (byte)(REVERSE[src[sPos] & 0xFF] & maskStart | dest[dPos] & ~maskStart);
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (dPos < dPosMax) {
                dest[dPos] = REVERSE[src[sPos] & 0xFF];
                ++dPos;
                ++sPos;
            }
            int cntFinish = (int)(count & 7L);
            if (cntFinish > 0) {
                int maskFinish = (1 << cntFinish) - 1;
                dest[dPos] = (byte)(REVERSE[src[sPos] & 0xFF] & maskFinish | dest[dPos] & ~maskFinish);
            }
        } else {
            int sNext;
            int shift = dPosRem - sPosRem;
            if (cntStart > 0) {
                int v;
                if (sPosRem + cntStart <= 8) {
                    if (shift > 0) {
                        sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
                        v = sNext << shift;
                    } else {
                        sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
                        v = sNext >>> -shift;
                    }
                    sPosRem += cntStart;
                } else {
                    sNext = REVERSE[src[sPos + 1] & 0xFF] & 0xFF;
                    v = (REVERSE[src[sPos] & 0xFF] & 0xFF) >>> -shift | sNext << 8 + shift;
                    ++sPos;
                    sPosRem = sPosRem + cntStart & 7;
                }
                dest[dPos] = (byte)(v & maskStart | dest[dPos] & ~maskStart);
                if ((count -= (long)cntStart) == 0L) {
                    return;
                }
                ++dPos;
            } else {
                if (count == 0L) {
                    return;
                }
                sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
            }
            int sPosRem8 = 8 - sPosRem;
            int dPosMax = dPos + (int)(count >>> 3);
            while (dPos < dPosMax) {
                int n = sNext >>> sPosRem;
                sNext = REVERSE[src[++sPos] & 0xFF] & 0xFF;
                dest[dPos] = (byte)(n | sNext << sPosRem8);
                ++dPos;
            }
            int cntFinish = (int)(count & 7L);
            if (cntFinish > 0) {
                int maskFinish = (1 << cntFinish) - 1;
                int v = sPosRem + cntFinish <= 8 ? sNext >>> sPosRem : sNext >>> sPosRem | (REVERSE[src[sPos + 1] & 0xFF] & 0xFF) << sPosRem8;
                dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void copyBitsFromNormalToReverseOrder(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> dPosRem + cntStart;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)(REVERSE[src[sPos] & 0xFF] & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = 65280 >>> cntFinish;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos] = (byte)(REVERSE[src[sPos] & 0xFF] & maskFinish | dest[dPos] & ~maskFinish);
                    // MONITOREXIT : byArray
                    return;
                }
                dest[dPos] = REVERSE[src[sPos] & 0xFF];
                ++dPos;
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
                    v = sNext >>> shift;
                } else {
                    sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
                    v = sNext << -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = REVERSE[src[sPos + 1] & 0xFF] & 0xFF;
                v = (REVERSE[src[sPos] & 0xFF] & 0xFF) << -shift | sNext >>> 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)(v & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = sNext << sPosRem;
            sNext = REVERSE[src[++sPos] & 0xFF] & 0xFF;
            dest[dPos] = (byte)(n | sNext >>> sPosRem8);
            ++dPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = 65280 >>> cntFinish;
        int v = sPosRem + cntFinish <= 8 ? sNext << sPosRem : sNext << sPosRem | (REVERSE[src[sPos + 1] & 0xFF] & 0xFF) >>> sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    public static void copyBitsFromNormalToReverseOrderNoSync(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> dPosRem + cntStart;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                dest[dPos] = (byte)(REVERSE[src[sPos] & 0xFF] & maskStart | dest[dPos] & ~maskStart);
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (dPos < dPosMax) {
                dest[dPos] = REVERSE[src[sPos] & 0xFF];
                ++dPos;
                ++sPos;
            }
            int cntFinish = (int)(count & 7L);
            if (cntFinish > 0) {
                int maskFinish = 65280 >>> cntFinish;
                dest[dPos] = (byte)(REVERSE[src[sPos] & 0xFF] & maskFinish | dest[dPos] & ~maskFinish);
            }
        } else {
            int sNext;
            int shift = dPosRem - sPosRem;
            if (cntStart > 0) {
                int v;
                if (sPosRem + cntStart <= 8) {
                    if (shift > 0) {
                        sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
                        v = sNext >>> shift;
                    } else {
                        sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
                        v = sNext << -shift;
                    }
                    sPosRem += cntStart;
                } else {
                    sNext = REVERSE[src[sPos + 1] & 0xFF] & 0xFF;
                    v = (REVERSE[src[sPos] & 0xFF] & 0xFF) << -shift | sNext >>> 8 + shift;
                    ++sPos;
                    sPosRem = sPosRem + cntStart & 7;
                }
                dest[dPos] = (byte)(v & maskStart | dest[dPos] & ~maskStart);
                if ((count -= (long)cntStart) == 0L) {
                    return;
                }
                ++dPos;
            } else {
                if (count == 0L) {
                    return;
                }
                sNext = REVERSE[src[sPos] & 0xFF] & 0xFF;
            }
            int sPosRem8 = 8 - sPosRem;
            int dPosMax = dPos + (int)(count >>> 3);
            while (dPos < dPosMax) {
                int n = sNext << sPosRem;
                sNext = REVERSE[src[++sPos] & 0xFF] & 0xFF;
                dest[dPos] = (byte)(n | sNext >>> sPosRem8);
                ++dPos;
            }
            int cntFinish = (int)(count & 7L);
            if (cntFinish > 0) {
                int maskFinish = 65280 >>> cntFinish;
                int v = sPosRem + cntFinish <= 8 ? sNext << sPosRem : sNext << sPosRem | (REVERSE[src[sPos + 1] & 0xFF] & 0xFF) >>> sPosRem8;
                dest[dPos] = (byte)(v & maskFinish | dest[dPos] & ~maskFinish);
            }
        }
    }

    public static byte[] packBits(boolean[] src, long srcPos, long count) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (srcPos > (long)src.length) {
            throw new IllegalArgumentException("The source position " + srcPos + " is out of the array boolean[" + src.length + "]");
        }
        if (count > (long)src.length - srcPos) {
            throw new IllegalArgumentException("Too short source array boolean[" + src.length + "]: it does not contain " + count + " bits since position " + srcPos);
        }
        assert (count == (long)((int)count));
        byte[] dest = new byte[(int)PackedBitArraysPer8.packedLength(count)];
        int sPos = (int)srcPos;
        int cnt = (int)(count >>> 3);
        for (int k = 0; k < cnt; ++k) {
            dest[k] = (byte)((src[sPos] ? 1 : 0) | (src[sPos + 1] ? 2 : 0) | (src[sPos + 2] ? 4 : 0) | (src[sPos + 3] ? 8 : 0) | (src[sPos + 4] ? 16 : 0) | (src[sPos + 5] ? 32 : 0) | (src[sPos + 6] ? 64 : 0) | (src[sPos + 7] ? 128 : 0));
            sPos += 8;
        }
        int countFinish = (int)count & 7;
        for (int k = 0; k < countFinish; ++k) {
            if (src[sPos]) {
                int n = cnt;
                dest[n] = (byte)(dest[n] | (byte)(1 << k));
            } else {
                int n = cnt;
                dest[n] = (byte)(dest[n] & (byte)(~(1 << k)));
            }
            ++sPos;
        }
        return dest;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void packBits(byte[] dest, long destPos, boolean[] src, int srcPos, int count) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (destPos & 7L) == 0L ? 0 : 8 - (int)(destPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        if (countStart > 0) {
            byte[] byArray = dest;
            // MONITORENTER : dest
            int srcPosMax = srcPos + countStart;
            while (srcPos < srcPosMax) {
                if (src[srcPos]) {
                    int n2 = (int)(destPos >>> 3);
                    dest[n2] = (byte)(dest[n2] | (byte)(1 << (int)(destPos & 7L)));
                } else {
                    int n3 = (int)(destPos >>> 3);
                    dest[n3] = (byte)(dest[n3] & (byte)(~(1 << (int)(destPos & 7L))));
                }
                ++srcPos;
                ++destPos;
            }
            // MONITOREXIT : byArray
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(destPos >>> 3); k < kMax; srcPos += 8, destPos += 8L, ++k) {
            dest[k] = (byte)((src[srcPos] ? 1 : 0) | (src[srcPos + 1] ? 2 : 0) | (src[srcPos + 2] ? 4 : 0) | (src[srcPos + 3] ? 8 : 0) | (src[srcPos + 4] ? 16 : 0) | (src[srcPos + 5] ? 32 : 0) | (src[srcPos + 6] ? 64 : 0) | (src[srcPos + 7] ? 128 : 0));
        }
        int countFinish = count & 7;
        if (countFinish <= 0) return;
        byte[] byArray = dest;
        // MONITORENTER : dest
        int srcPosMax = srcPos + countFinish;
        while (true) {
            if (srcPos >= srcPosMax) {
                // MONITOREXIT : byArray
                return;
            }
            if (src[srcPos]) {
                int n4 = (int)(destPos >>> 3);
                dest[n4] = (byte)(dest[n4] | (byte)(1 << (int)(destPos & 7L)));
            } else {
                int n5 = (int)(destPos >>> 3);
                dest[n5] = (byte)(dest[n5] & (byte)(~(1 << (int)(destPos & 7L))));
            }
            ++srcPos;
            ++destPos;
        }
    }

    public static void packBitsNoSync(byte[] dest, long destPos, boolean[] src, int srcPos, int count) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (destPos & 7L) == 0L ? 0 : 8 - (int)(destPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        if (countStart > 0) {
            int srcPosMax = srcPos + countStart;
            while (srcPos < srcPosMax) {
                if (src[srcPos]) {
                    int n2 = (int)(destPos >>> 3);
                    dest[n2] = (byte)(dest[n2] | (byte)(1 << (int)(destPos & 7L)));
                } else {
                    int n3 = (int)(destPos >>> 3);
                    dest[n3] = (byte)(dest[n3] & (byte)(~(1 << (int)(destPos & 7L))));
                }
                ++srcPos;
                ++destPos;
            }
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(destPos >>> 3); k < kMax; ++k) {
            dest[k] = (byte)((src[srcPos] ? 1 : 0) | (src[srcPos + 1] ? 2 : 0) | (src[srcPos + 2] ? 4 : 0) | (src[srcPos + 3] ? 8 : 0) | (src[srcPos + 4] ? 16 : 0) | (src[srcPos + 5] ? 32 : 0) | (src[srcPos + 6] ? 64 : 0) | (src[srcPos + 7] ? 128 : 0));
            srcPos += 8;
            destPos += 8L;
        }
        int countFinish = count & 7;
        if (countFinish > 0) {
            int srcPosMax = srcPos + countFinish;
            while (srcPos < srcPosMax) {
                if (src[srcPos]) {
                    int n4 = (int)(destPos >>> 3);
                    dest[n4] = (byte)(dest[n4] | (byte)(1 << (int)(destPos & 7L)));
                } else {
                    int n5 = (int)(destPos >>> 3);
                    dest[n5] = (byte)(dest[n5] & (byte)(~(1 << (int)(destPos & 7L))));
                }
                ++srcPos;
                ++destPos;
            }
        }
    }

    public static byte[] packBitsInReverseOrder(boolean[] src, long srcPos, long count) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (srcPos > (long)src.length) {
            throw new IllegalArgumentException("The source position " + srcPos + " is out of the array boolean[" + src.length + "]");
        }
        if (count > (long)src.length - srcPos) {
            throw new IllegalArgumentException("Too short source array boolean[" + src.length + "]: it does not contain " + count + " bits since position " + srcPos);
        }
        assert (count == (long)((int)count));
        byte[] dest = new byte[(int)PackedBitArraysPer8.packedLength(count)];
        int sPos = (int)srcPos;
        int cnt = (int)(count >>> 3);
        for (int k = 0; k < cnt; ++k) {
            dest[k] = (byte)((src[sPos] ? 128 : 0) | (src[sPos + 1] ? 64 : 0) | (src[sPos + 2] ? 32 : 0) | (src[sPos + 3] ? 16 : 0) | (src[sPos + 4] ? 8 : 0) | (src[sPos + 5] ? 4 : 0) | (src[sPos + 6] ? 2 : 0) | (src[sPos + 7] ? 1 : 0));
            sPos += 8;
        }
        int countFinish = (int)count & 7;
        for (int k = 0; k < countFinish; ++k) {
            if (src[sPos]) {
                int n = cnt;
                dest[n] = (byte)(dest[n] | (byte)(1 << 7 - k));
            } else {
                int n = cnt;
                dest[n] = (byte)(dest[n] & (byte)(~(1 << 7 - k)));
            }
            ++sPos;
        }
        return dest;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void packBitsInReverseOrder(byte[] dest, long destPos, boolean[] src, int srcPos, int count) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (destPos & 7L) == 0L ? 0 : 8 - (int)(destPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        if (countStart > 0) {
            byte[] byArray = dest;
            // MONITORENTER : dest
            int srcPosMax = srcPos + countStart;
            while (srcPos < srcPosMax) {
                if (src[srcPos]) {
                    int n2 = (int)(destPos >>> 3);
                    dest[n2] = (byte)(dest[n2] | (byte)(1 << (int)(7L - (destPos & 7L))));
                } else {
                    int n3 = (int)(destPos >>> 3);
                    dest[n3] = (byte)(dest[n3] & (byte)(~(1 << (int)(7L - (destPos & 7L)))));
                }
                ++srcPos;
                ++destPos;
            }
            // MONITOREXIT : byArray
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(destPos >>> 3); k < kMax; srcPos += 8, destPos += 8L, ++k) {
            dest[k] = (byte)((src[srcPos] ? 128 : 0) | (src[srcPos + 1] ? 64 : 0) | (src[srcPos + 2] ? 32 : 0) | (src[srcPos + 3] ? 16 : 0) | (src[srcPos + 4] ? 8 : 0) | (src[srcPos + 5] ? 4 : 0) | (src[srcPos + 6] ? 2 : 0) | (src[srcPos + 7] ? 1 : 0));
        }
        int countFinish = count & 7;
        if (countFinish <= 0) return;
        byte[] byArray = dest;
        // MONITORENTER : dest
        int srcPosMax = srcPos + countFinish;
        while (true) {
            if (srcPos >= srcPosMax) {
                // MONITOREXIT : byArray
                return;
            }
            if (src[srcPos]) {
                int n4 = (int)(destPos >>> 3);
                dest[n4] = (byte)(dest[n4] | (byte)(1 << (int)(7L - (destPos & 7L))));
            } else {
                int n5 = (int)(destPos >>> 3);
                dest[n5] = (byte)(dest[n5] & (byte)(~(1 << (int)(7L - (destPos & 7L)))));
            }
            ++srcPos;
            ++destPos;
        }
    }

    public static void packBitsInReverseOrderNoSync(byte[] dest, long destPos, boolean[] src, int srcPos, int count) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (destPos & 7L) == 0L ? 0 : 8 - (int)(destPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        if (countStart > 0) {
            int srcPosMax = srcPos + countStart;
            while (srcPos < srcPosMax) {
                if (src[srcPos]) {
                    int n2 = (int)(destPos >>> 3);
                    dest[n2] = (byte)(dest[n2] | (byte)(1 << (int)(7L - (destPos & 7L))));
                } else {
                    int n3 = (int)(destPos >>> 3);
                    dest[n3] = (byte)(dest[n3] & (byte)(~(1 << (int)(7L - (destPos & 7L)))));
                }
                ++srcPos;
                ++destPos;
            }
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(destPos >>> 3); k < kMax; ++k) {
            dest[k] = (byte)((src[srcPos] ? 128 : 0) | (src[srcPos + 1] ? 64 : 0) | (src[srcPos + 2] ? 32 : 0) | (src[srcPos + 3] ? 16 : 0) | (src[srcPos + 4] ? 8 : 0) | (src[srcPos + 5] ? 4 : 0) | (src[srcPos + 6] ? 2 : 0) | (src[srcPos + 7] ? 1 : 0));
            srcPos += 8;
            destPos += 8L;
        }
        int countFinish = count & 7;
        if (countFinish > 0) {
            int srcPosMax = srcPos + countFinish;
            while (srcPos < srcPosMax) {
                if (src[srcPos]) {
                    int n4 = (int)(destPos >>> 3);
                    dest[n4] = (byte)(dest[n4] | (byte)(1 << (int)(7L - (destPos & 7L))));
                } else {
                    int n5 = (int)(destPos >>> 3);
                    dest[n5] = (byte)(dest[n5] & (byte)(~(1 << (int)(7L - (destPos & 7L)))));
                }
                ++srcPos;
                ++destPos;
            }
        }
    }

    public static boolean[] unpackBits(byte[] src, long srcPos, long count) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        boolean[] result = new boolean[(int)count];
        PackedBitArraysPer8.unpackBits(result, 0, src, srcPos, result.length);
        return result;
    }

    public static void unpackBits(boolean[] dest, int destPos, byte[] src, long srcPos, int count) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 1) != 0;
            dest[destPos + 1] = (v & 2) != 0;
            dest[destPos + 2] = (v & 4) != 0;
            dest[destPos + 3] = (v & 8) != 0;
            dest[destPos + 4] = (v & 0x10) != 0;
            dest[destPos + 5] = (v & 0x20) != 0;
            dest[destPos + 6] = (v & 0x40) != 0;
            dest[destPos + 7] = (v & 0x80) != 0;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0;
            ++srcPos;
            ++destPos;
        }
    }

    public static boolean[] unpackBitsInReverseOrder(byte[] src, long srcPos, long count) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        boolean[] result = new boolean[(int)count];
        PackedBitArraysPer8.unpackBitsInReverseOrder(result, 0, src, srcPos, result.length);
        return result;
    }

    public static void unpackBitsInReverseOrder(boolean[] dest, int destPos, byte[] src, long srcPos, int count) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 0x80) != 0;
            dest[destPos + 1] = (v & 0x40) != 0;
            dest[destPos + 2] = (v & 0x20) != 0;
            dest[destPos + 3] = (v & 0x10) != 0;
            dest[destPos + 4] = (v & 8) != 0;
            dest[destPos + 5] = (v & 4) != 0;
            dest[destPos + 6] = (v & 2) != 0;
            dest[destPos + 7] = (v & 1) != 0;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0;
            ++srcPos;
            ++destPos;
        }
    }

    public static boolean[] unpackBits(byte[] src, long srcPos, long count, boolean bit0Value, boolean bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        boolean[] result = new boolean[(int)count];
        PackedBitArraysPer8.unpackBits(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBits(boolean[] dest, int destPos, byte[] src, long srcPos, int count, boolean bit0Value, boolean bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 1) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static boolean[] unpackBitsInReverseOrder(byte[] src, long srcPos, long count, boolean bit0Value, boolean bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        boolean[] result = new boolean[(int)count];
        PackedBitArraysPer8.unpackBitsInReverseOrder(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBitsInReverseOrder(boolean[] dest, int destPos, byte[] src, long srcPos, int count, boolean bit0Value, boolean bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 1) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static char[] unpackBitsToChars(byte[] src, long srcPos, long count, char bit0Value, char bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        char[] result = new char[(int)count];
        PackedBitArraysPer8.unpackBits(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBits(char[] dest, int destPos, byte[] src, long srcPos, int count, char bit0Value, char bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 1) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static char[] unpackBitsInReverseOrderToChars(byte[] src, long srcPos, long count, char bit0Value, char bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        char[] result = new char[(int)count];
        PackedBitArraysPer8.unpackBitsInReverseOrder(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBitsInReverseOrder(char[] dest, int destPos, byte[] src, long srcPos, int count, char bit0Value, char bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 1) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static byte[] unpackBitsToBytes(byte[] src, long srcPos, long count, byte bit0Value, byte bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        byte[] result = new byte[(int)count];
        PackedBitArraysPer8.unpackBits(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBits(byte[] dest, int destPos, byte[] src, long srcPos, int count, byte bit0Value, byte bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 1) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static byte[] unpackBitsInReverseOrderToBytes(byte[] src, long srcPos, long count, byte bit0Value, byte bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        byte[] result = new byte[(int)count];
        PackedBitArraysPer8.unpackBitsInReverseOrder(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBitsInReverseOrder(byte[] dest, int destPos, byte[] src, long srcPos, int count, byte bit0Value, byte bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 1) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static short[] unpackBitsToShorts(byte[] src, long srcPos, long count, short bit0Value, short bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        short[] result = new short[(int)count];
        PackedBitArraysPer8.unpackBits(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBits(short[] dest, int destPos, byte[] src, long srcPos, int count, short bit0Value, short bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 1) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static short[] unpackBitsInReverseOrderToShorts(byte[] src, long srcPos, long count, short bit0Value, short bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        short[] result = new short[(int)count];
        PackedBitArraysPer8.unpackBitsInReverseOrder(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBitsInReverseOrder(short[] dest, int destPos, byte[] src, long srcPos, int count, short bit0Value, short bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 1) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static int[] unpackBitsToInts(byte[] src, long srcPos, long count, int bit0Value, int bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        int[] result = new int[(int)count];
        PackedBitArraysPer8.unpackBits(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBits(int[] dest, int destPos, byte[] src, long srcPos, int count, int bit0Value, int bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 1) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static int[] unpackBitsInReverseOrderToInts(byte[] src, long srcPos, long count, int bit0Value, int bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        int[] result = new int[(int)count];
        PackedBitArraysPer8.unpackBitsInReverseOrder(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBitsInReverseOrder(int[] dest, int destPos, byte[] src, long srcPos, int count, int bit0Value, int bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 1) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static long[] unpackBitsToLongs(byte[] src, long srcPos, long count, long bit0Value, long bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        long[] result = new long[(int)count];
        PackedBitArraysPer8.unpackBits(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBits(long[] dest, int destPos, byte[] src, long srcPos, int count, long bit0Value, long bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 1) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static long[] unpackBitsInReverseOrderToLongs(byte[] src, long srcPos, long count, long bit0Value, long bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        long[] result = new long[(int)count];
        PackedBitArraysPer8.unpackBitsInReverseOrder(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBitsInReverseOrder(long[] dest, int destPos, byte[] src, long srcPos, int count, long bit0Value, long bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 1) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static float[] unpackBitsToFloats(byte[] src, long srcPos, long count, float bit0Value, float bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        float[] result = new float[(int)count];
        PackedBitArraysPer8.unpackBits(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBits(float[] dest, int destPos, byte[] src, long srcPos, int count, float bit0Value, float bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 1) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static float[] unpackBitsInReverseOrderToFloats(byte[] src, long srcPos, long count, float bit0Value, float bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        float[] result = new float[(int)count];
        PackedBitArraysPer8.unpackBitsInReverseOrder(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBitsInReverseOrder(float[] dest, int destPos, byte[] src, long srcPos, int count, float bit0Value, float bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 1) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static double[] unpackBitsToDoubles(byte[] src, long srcPos, long count, double bit0Value, double bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        double[] result = new double[(int)count];
        PackedBitArraysPer8.unpackBits(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBits(double[] dest, int destPos, byte[] src, long srcPos, int count, double bit0Value, double bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 1) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(srcPos & 7L)) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static double[] unpackBitsInReverseOrderToDoubles(byte[] src, long srcPos, long count, double bit0Value, double bit1Value) {
        Objects.requireNonNull(src, "Null src");
        if (srcPos < 0L) {
            throw new IllegalArgumentException("Negative srcPos = " + srcPos);
        }
        if (count < 0L) {
            throw new IllegalArgumentException("Negative count = " + count);
        }
        if (count > PackedBitArraysPer8.unpackedLength(src) - srcPos) {
            throw new IllegalArgumentException("Too short source array byte[" + src.length + "]: it cannot contain " + count + " bits since position " + srcPos);
        }
        if (count > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large bit array for unpacking to Java array: " + count + " >= 2^31 bits");
        }
        double[] result = new double[(int)count];
        PackedBitArraysPer8.unpackBitsInReverseOrder(result, 0, src, srcPos, result.length, bit0Value, bit1Value);
        return result;
    }

    public static void unpackBitsInReverseOrder(double[] dest, int destPos, byte[] src, long srcPos, int count, double bit0Value, double bit1Value) {
        int k;
        int countStart;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int n = countStart = (srcPos & 7L) == 0L ? 0 : 8 - (int)(srcPos & 7L);
        if (countStart > count) {
            countStart = count;
        }
        int destPosMax = destPos + countStart;
        while (destPos < destPosMax) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
        int cnt = (count -= countStart) >>> 3;
        int kMax = k + cnt;
        for (k = (int)(srcPos >>> 3); k < kMax; ++k) {
            byte v = src[k];
            srcPos += 8L;
            dest[destPos] = (v & 0x80) != 0 ? bit1Value : bit0Value;
            dest[destPos + 1] = (v & 0x40) != 0 ? bit1Value : bit0Value;
            dest[destPos + 2] = (v & 0x20) != 0 ? bit1Value : bit0Value;
            dest[destPos + 3] = (v & 0x10) != 0 ? bit1Value : bit0Value;
            dest[destPos + 4] = (v & 8) != 0 ? bit1Value : bit0Value;
            dest[destPos + 5] = (v & 4) != 0 ? bit1Value : bit0Value;
            dest[destPos + 6] = (v & 2) != 0 ? bit1Value : bit0Value;
            dest[destPos + 7] = (v & 1) != 0 ? bit1Value : bit0Value;
            destPos += 8;
        }
        int countFinish = count & 7;
        int destPosMax2 = destPos + countFinish;
        while (destPos < destPosMax2) {
            dest[destPos] = (src[(int)(srcPos >>> 3)] & 1 << (int)(7L - (srcPos & 7L))) != 0 ? bit1Value : bit0Value;
            ++srcPos;
            ++destPos;
        }
    }

    public static void notBits(byte[] dest, long destPos, long count) {
        PackedBitArraysPer8.notBits(dest, destPos, dest, destPos, count);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void notBits(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)(~(src[sPos] & 0xFF) & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = (1 << cntFinish) - 1;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos] = (byte)(~(src[sPos] & 0xFF) & maskFinish | dest[dPos] & ~maskFinish);
                    // MONITOREXIT : byArray
                    return;
                }
                dest[dPos] = (byte)(~(src[sPos] & 0xFF));
                ++dPos;
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) >>> -shift | sNext << 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)(~v & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = sNext >>> sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[dPos] = (byte)(~(n | sNext << sPosRem8));
            ++dPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = (1 << cntFinish) - 1;
        int v = sPosRem + cntFinish <= 8 ? sNext >>> sPosRem : sNext >>> sPosRem | (src[sPos + 1] & 0xFF) << sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)(~v & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    public static void notBitsInReverseOrder(byte[] dest, long destPos, long count) {
        PackedBitArraysPer8.notBitsInReverseOrder(dest, destPos, dest, destPos, count);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void notBitsInReverseOrder(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> (int)((long)dPosRem + count);
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)(~(src[sPos] & 0xFF) & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = 65280 >>> cntFinish;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos] = (byte)(~(src[sPos] & 0xFF) & maskFinish | dest[dPos] & ~maskFinish);
                    // MONITOREXIT : byArray
                    return;
                }
                dest[dPos] = (byte)(~(src[sPos] & 0xFF));
                ++dPos;
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) << -shift | sNext >>> 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)(~v & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = sNext << sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[dPos] = (byte)(~(n | sNext >>> sPosRem8));
            ++dPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = 65280 >>> cntFinish;
        int v = sPosRem + cntFinish <= 8 ? sNext << sPosRem : sNext << sPosRem | (src[sPos + 1] & 0xFF) >>> sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)(~v & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void andBits(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                int n = dPos++;
                dest[n] = (byte)(dest[n] & (byte)(src[sPos] & 0xFF | ~maskStart));
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = (1 << cntFinish) - 1;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    int n = dPos;
                    dest[n] = (byte)(dest[n] & (byte)(src[sPos] & 0xFF | ~maskFinish));
                    // MONITOREXIT : byArray
                    return;
                }
                int n = dPos++;
                dest[n] = (byte)(dest[n] & (byte)(src[sPos] & 0xFF));
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) >>> -shift | sNext << 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            int n = dPos++;
            dest[n] = (byte)(dest[n] & (byte)(v | ~maskStart));
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = dPos++;
            int n2 = sNext >>> sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[n] = (byte)(dest[n] & (byte)(n2 | sNext << sPosRem8));
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = (1 << cntFinish) - 1;
        int v = sPosRem + cntFinish <= 8 ? sNext >>> sPosRem : sNext >>> sPosRem | (src[sPos + 1] & 0xFF) << sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        int n = dPos;
        dest[n] = (byte)(dest[n] & (byte)(v | ~maskFinish));
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void andBitsInReverseOrder(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> (int)((long)dPosRem + count);
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                int n = dPos++;
                dest[n] = (byte)(dest[n] & (byte)(src[sPos] & 0xFF | ~maskStart));
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = 65280 >>> cntFinish;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    int n = dPos;
                    dest[n] = (byte)(dest[n] & (byte)(src[sPos] & 0xFF | ~maskFinish));
                    // MONITOREXIT : byArray
                    return;
                }
                int n = dPos++;
                dest[n] = (byte)(dest[n] & (byte)(src[sPos] & 0xFF));
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) << -shift | sNext >>> 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            int n = dPos++;
            dest[n] = (byte)(dest[n] & (byte)(v | ~maskStart));
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = dPos++;
            int n2 = sNext << sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[n] = (byte)(dest[n] & (byte)(n2 | sNext >>> sPosRem8));
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = 65280 >>> cntFinish;
        int v = sPosRem + cntFinish <= 8 ? sNext << sPosRem : sNext << sPosRem | (src[sPos + 1] & 0xFF) >>> sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        int n = dPos;
        dest[n] = (byte)(dest[n] & (byte)(v | ~maskFinish));
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void orBits(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                int n = dPos++;
                dest[n] = (byte)(dest[n] | (byte)(src[sPos] & 0xFF & maskStart));
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = (1 << cntFinish) - 1;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    int n = dPos;
                    dest[n] = (byte)(dest[n] | (byte)(src[sPos] & 0xFF & maskFinish));
                    // MONITOREXIT : byArray
                    return;
                }
                int n = dPos++;
                dest[n] = (byte)(dest[n] | (byte)(src[sPos] & 0xFF));
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) >>> -shift | sNext << 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            int n = dPos++;
            dest[n] = (byte)(dest[n] | (byte)(v & maskStart));
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = dPos++;
            int n2 = sNext >>> sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[n] = (byte)(dest[n] | (byte)(n2 | sNext << sPosRem8));
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = (1 << cntFinish) - 1;
        int v = sPosRem + cntFinish <= 8 ? sNext >>> sPosRem : sNext >>> sPosRem | (src[sPos + 1] & 0xFF) << sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        int n = dPos;
        dest[n] = (byte)(dest[n] | (byte)(v & maskFinish));
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void orBitsInReverseOrder(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> (int)((long)dPosRem + count);
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                int n = dPos++;
                dest[n] = (byte)(dest[n] | (byte)(src[sPos] & 0xFF & maskStart));
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = 65280 >>> cntFinish;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    int n = dPos;
                    dest[n] = (byte)(dest[n] | (byte)(src[sPos] & 0xFF & maskFinish));
                    // MONITOREXIT : byArray
                    return;
                }
                int n = dPos++;
                dest[n] = (byte)(dest[n] | (byte)(src[sPos] & 0xFF));
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) << -shift | sNext >>> 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            int n = dPos++;
            dest[n] = (byte)(dest[n] | (byte)(v & maskStart));
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = dPos++;
            int n2 = sNext << sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[n] = (byte)(dest[n] | (byte)(n2 | sNext >>> sPosRem8));
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = 65280 >>> cntFinish;
        int v = sPosRem + cntFinish <= 8 ? sNext << sPosRem : sNext << sPosRem | (src[sPos + 1] & 0xFF) >>> sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        int n = dPos;
        dest[n] = (byte)(dest[n] | (byte)(v & maskFinish));
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void xorBits(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)((dest[dPos] ^ src[sPos] & 0xFF) & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = (1 << cntFinish) - 1;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos] = (byte)((dest[dPos] ^ src[sPos] & 0xFF) & maskFinish | dest[dPos] & ~maskFinish);
                    // MONITOREXIT : byArray
                    return;
                }
                int n = dPos++;
                dest[n] = (byte)(dest[n] ^ (byte)(src[sPos] & 0xFF));
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) >>> -shift | sNext << 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)((dest[dPos] ^ v) & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = dPos++;
            int n2 = sNext >>> sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[n] = (byte)(dest[n] ^ (byte)(n2 | sNext << sPosRem8));
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = (1 << cntFinish) - 1;
        int v = sPosRem + cntFinish <= 8 ? sNext >>> sPosRem : sNext >>> sPosRem | (src[sPos + 1] & 0xFF) << sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)((dest[dPos] ^ v) & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void xorBitsInReverseOrder(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> (int)((long)dPosRem + count);
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)((dest[dPos] ^ src[sPos] & 0xFF) & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = 65280 >>> cntFinish;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos] = (byte)((dest[dPos] ^ src[sPos] & 0xFF) & maskFinish | dest[dPos] & ~maskFinish);
                    // MONITOREXIT : byArray
                    return;
                }
                int n = dPos++;
                dest[n] = (byte)(dest[n] ^ (byte)(src[sPos] & 0xFF));
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) << -shift | sNext >>> 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)((dest[dPos] ^ v) & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = dPos++;
            int n2 = sNext << sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[n] = (byte)(dest[n] ^ (byte)(n2 | sNext >>> sPosRem8));
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = 65280 >>> cntFinish;
        int v = sPosRem + cntFinish <= 8 ? sNext << sPosRem : sNext << sPosRem | (src[sPos + 1] & 0xFF) >>> sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)((dest[dPos] ^ v) & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void andNotBits(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)(dest[dPos] & ~(src[sPos] & 0xFF) & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = (1 << cntFinish) - 1;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos] = (byte)(dest[dPos] & ~(src[sPos] & 0xFF) & maskFinish | dest[dPos] & ~maskFinish);
                    // MONITOREXIT : byArray
                    return;
                }
                int n = dPos++;
                dest[n] = (byte)(dest[n] & (byte)(~(src[sPos] & 0xFF)));
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) >>> -shift | sNext << 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)(dest[dPos] & ~v & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = dPos++;
            int n2 = sNext >>> sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[n] = (byte)(dest[n] & (byte)(~(n2 | sNext << sPosRem8)));
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = (1 << cntFinish) - 1;
        int v = sPosRem + cntFinish <= 8 ? sNext >>> sPosRem : sNext >>> sPosRem | (src[sPos + 1] & 0xFF) << sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)(dest[dPos] & ~v & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void andNotBitsInReverseOrder(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> (int)((long)dPosRem + count);
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)(dest[dPos] & ~(src[sPos] & 0xFF) & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = 65280 >>> cntFinish;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos] = (byte)(dest[dPos] & ~(src[sPos] & 0xFF) & maskFinish | dest[dPos] & ~maskFinish);
                    // MONITOREXIT : byArray
                    return;
                }
                int n = dPos++;
                dest[n] = (byte)(dest[n] & (byte)(~(src[sPos] & 0xFF)));
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) << -shift | sNext >>> 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)(dest[dPos] & ~v & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = dPos++;
            int n2 = sNext << sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[n] = (byte)(dest[n] & (byte)(~(n2 | sNext >>> sPosRem8)));
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = 65280 >>> cntFinish;
        int v = sPosRem + cntFinish <= 8 ? sNext << sPosRem : sNext << sPosRem | (src[sPos + 1] & 0xFF) >>> sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)(dest[dPos] & ~v & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void orNotBits(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)((dest[dPos] | ~(src[sPos] & 0xFF)) & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = (1 << cntFinish) - 1;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos] = (byte)((dest[dPos] | ~(src[sPos] & 0xFF)) & maskFinish | dest[dPos] & ~maskFinish);
                    // MONITOREXIT : byArray
                    return;
                }
                int n = dPos++;
                dest[n] = (byte)(dest[n] | (byte)(~(src[sPos] & 0xFF)));
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) >>> -shift | sNext << 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)((dest[dPos] | ~v) & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = dPos++;
            int n2 = sNext >>> sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[n] = (byte)(dest[n] | (byte)(~(n2 | sNext << sPosRem8)));
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = (1 << cntFinish) - 1;
        int v = sPosRem + cntFinish <= 8 ? sNext >>> sPosRem : sNext >>> sPosRem | (src[sPos + 1] & 0xFF) << sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)((dest[dPos] | ~v) & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void orNotBitsInReverseOrder(byte[] dest, long destPos, byte[] src, long srcPos, long count) {
        int sNext;
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        int sPos = (int)(srcPos >>> 3);
        int dPos = (int)(destPos >>> 3);
        int sPosRem = (int)(srcPos & 7L);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> (int)((long)dPosRem + count);
        }
        if (sPosRem == dPosRem) {
            if (cntStart > 0) {
                byte[] byArray = dest;
                // MONITORENTER : dest
                dest[dPos] = (byte)((dest[dPos] | ~(src[sPos] & 0xFF)) & maskStart | dest[dPos] & ~maskStart);
                // MONITOREXIT : byArray
                count -= (long)cntStart;
                ++dPos;
                ++sPos;
            }
            int dPosMax = dPos + (int)(count >>> 3);
            while (true) {
                if (dPos >= dPosMax) {
                    int cntFinish = (int)(count & 7L);
                    if (cntFinish <= 0) return;
                    int maskFinish = 65280 >>> cntFinish;
                    byte[] byArray = dest;
                    // MONITORENTER : dest
                    dest[dPos] = (byte)((dest[dPos] | ~(src[sPos] & 0xFF)) & maskFinish | dest[dPos] & ~maskFinish);
                    // MONITOREXIT : byArray
                    return;
                }
                int n = dPos++;
                dest[n] = (byte)(dest[n] | (byte)(~(src[sPos] & 0xFF)));
                ++sPos;
            }
        }
        int shift = dPosRem - sPosRem;
        if (cntStart > 0) {
            int v;
            if (sPosRem + cntStart <= 8) {
                if (shift > 0) {
                    sNext = src[sPos] & 0xFF;
                    v = sNext >>> shift;
                } else {
                    sNext = src[sPos] & 0xFF;
                    v = sNext << -shift;
                }
                sPosRem += cntStart;
            } else {
                sNext = src[sPos + 1] & 0xFF;
                v = (src[sPos] & 0xFF) << -shift | sNext >>> 8 + shift;
                ++sPos;
                sPosRem = sPosRem + cntStart & 7;
            }
            byte[] byArray = dest;
            // MONITORENTER : dest
            dest[dPos] = (byte)((dest[dPos] | ~v) & maskStart | dest[dPos] & ~maskStart);
            // MONITOREXIT : byArray
            if ((count -= (long)cntStart) == 0L) {
                return;
            }
            ++dPos;
        } else {
            if (count == 0L) {
                return;
            }
            sNext = src[sPos] & 0xFF;
        }
        int sPosRem8 = 8 - sPosRem;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            int n = dPos++;
            int n2 = sNext << sPosRem;
            sNext = src[++sPos] & 0xFF;
            dest[n] = (byte)(dest[n] | (byte)(~(n2 | sNext >>> sPosRem8)));
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = 65280 >>> cntFinish;
        int v = sPosRem + cntFinish <= 8 ? sNext << sPosRem : sNext << sPosRem | (src[sPos + 1] & 0xFF) >>> sPosRem8;
        byte[] byArray = dest;
        // MONITORENTER : dest
        dest[dPos] = (byte)((dest[dPos] | ~v) & maskFinish | dest[dPos] & ~maskFinish);
        // MONITOREXIT : byArray
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void fillBits(byte[] dest, long destPos, long count, boolean value) {
        Objects.requireNonNull(dest, "Null dest");
        int dPos = (int)(destPos >>> 3);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (cntStart > 0) {
            byte[] byArray = dest;
            // MONITORENTER : dest
            if (value) {
                int n = dPos;
                dest[n] = (byte)(dest[n] | (byte)maskStart);
            } else {
                int n = dPos;
                dest[n] = (byte)(dest[n] & (byte)(~maskStart));
            }
            // MONITOREXIT : byArray
            count -= (long)cntStart;
            ++dPos;
        }
        int byteValue = value ? -1 : 0;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            dest[dPos] = byteValue;
            ++dPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = (1 << cntFinish) - 1;
        byte[] byArray = dest;
        // MONITORENTER : dest
        if (value) {
            int n = dPos;
            dest[n] = (byte)(dest[n] | (byte)maskFinish);
            return;
        }
        int n = dPos;
        dest[n] = (byte)(dest[n] & (byte)(~maskFinish));
        // MONITOREXIT : byArray
    }

    public static void fillBitsNoSync(byte[] dest, long destPos, long count, boolean value) {
        Objects.requireNonNull(dest, "Null dest");
        int dPos = (int)(destPos >>> 3);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = -1 << dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << dPosRem + cntStart) - 1;
        }
        if (cntStart > 0) {
            if (value) {
                int n = dPos;
                dest[n] = (byte)(dest[n] | (byte)maskStart);
            } else {
                int n = dPos;
                dest[n] = (byte)(dest[n] & (byte)(~maskStart));
            }
            count -= (long)cntStart;
            ++dPos;
        }
        int byteValue = value ? -1 : 0;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            dest[dPos] = byteValue;
            ++dPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish > 0) {
            int maskFinish = (1 << cntFinish) - 1;
            if (value) {
                int n = dPos;
                dest[n] = (byte)(dest[n] | (byte)maskFinish);
            } else {
                int n = dPos;
                dest[n] = (byte)(dest[n] & (byte)(~maskFinish));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static void fillBitsInReverseOrder(byte[] dest, long destPos, long count, boolean value) {
        Objects.requireNonNull(dest, "Null dest");
        int dPos = (int)(destPos >>> 3);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> (int)((long)dPosRem + count);
        }
        if (cntStart > 0) {
            byte[] byArray = dest;
            // MONITORENTER : dest
            if (value) {
                int n = dPos;
                dest[n] = (byte)(dest[n] | (byte)maskStart);
            } else {
                int n = dPos;
                dest[n] = (byte)(dest[n] & (byte)(~maskStart));
            }
            // MONITOREXIT : byArray
            count -= (long)cntStart;
            ++dPos;
        }
        int byteValue = value ? -1 : 0;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            dest[dPos] = byteValue;
            ++dPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish <= 0) return;
        int maskFinish = 65280 >>> cntFinish;
        byte[] byArray = dest;
        // MONITORENTER : dest
        if (value) {
            int n = dPos;
            dest[n] = (byte)(dest[n] | (byte)maskFinish);
            return;
        }
        int n = dPos;
        dest[n] = (byte)(dest[n] & (byte)(~maskFinish));
        // MONITOREXIT : byArray
    }

    public static void fillBitsInReverseOrderNoSync(byte[] dest, long destPos, long count, boolean value) {
        Objects.requireNonNull(dest, "Null dest");
        int dPos = (int)(destPos >>> 3);
        int dPosRem = (int)(destPos & 7L);
        int cntStart = -dPosRem & 7;
        int maskStart = 255 >>> dPosRem;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= 65280 >>> (int)((long)dPosRem + count);
        }
        if (cntStart > 0) {
            if (value) {
                int n = dPos;
                dest[n] = (byte)(dest[n] | (byte)maskStart);
            } else {
                int n = dPos;
                dest[n] = (byte)(dest[n] & (byte)(~maskStart));
            }
            count -= (long)cntStart;
            ++dPos;
        }
        int byteValue = value ? -1 : 0;
        int dPosMax = dPos + (int)(count >>> 3);
        while (dPos < dPosMax) {
            dest[dPos] = byteValue;
            ++dPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish > 0) {
            int maskFinish = 65280 >>> cntFinish;
            if (value) {
                int n = dPos;
                dest[n] = (byte)(dest[n] | (byte)maskFinish);
            } else {
                int n = dPos;
                dest[n] = (byte)(dest[n] & (byte)(~maskFinish));
            }
        }
    }

    public static long cardinality(byte[] src, long fromIndex, long toIndex) {
        Objects.requireNonNull(src, "Null src argument in cardinality method");
        if (fromIndex < 0L) {
            throw new ArrayIndexOutOfBoundsException("Bit array index out of range: initial index = " + fromIndex);
        }
        if (toIndex > (long)src.length << 3) {
            throw new ArrayIndexOutOfBoundsException("Bit array index out of range: end index = " + toIndex);
        }
        if (fromIndex > toIndex) {
            throw new ArrayIndexOutOfBoundsException("Bit array index out of range: initial index = " + fromIndex + " > end index = " + toIndex);
        }
        long count = toIndex - fromIndex;
        int sPos = (int)(fromIndex >>> 3);
        int sPosRem = (int)(fromIndex & 7L);
        int cntStart = -sPosRem & 7;
        int maskStart = -1 << sPosRem & 0xFF;
        if ((long)cntStart > count) {
            cntStart = (int)count;
            maskStart &= (1 << sPosRem + cntStart) - 1;
        }
        long result = 0L;
        if (cntStart > 0) {
            result += (long)Integer.bitCount(src[sPos] & maskStart);
            count -= (long)cntStart;
            ++sPos;
        }
        int sPosMax = sPos + (int)(count >>> 3);
        while (sPos < sPosMax) {
            result += (long)Integer.bitCount(src[sPos] & 0xFF);
            ++sPos;
        }
        int cntFinish = (int)(count & 7L);
        if (cntFinish > 0) {
            int maskFinish = (1 << cntFinish) - 1 & 0xFF;
            result += (long)Integer.bitCount(src[sPos] & maskFinish);
        }
        return result;
    }

    public static void reverseBitOrderInPlace(byte[] bytes) {
        Objects.requireNonNull(bytes, "Null bytes");
        for (int i = 0; i < bytes.length; ++i) {
            bytes[i] = REVERSE[bytes[i] & 0xFF];
        }
    }

    public static byte[] reverseBitOrder(byte[] src) {
        return PackedBitArraysPer8.reverseBitOrder(src, 0, src.length);
    }

    public static byte[] reverseBitOrder(byte[] src, int srcPos, int count) {
        Objects.requireNonNull(src, "Null src");
        JArrays.rangeCheck(src.length, srcPos, count);
        byte[] result = new byte[count];
        int i = srcPos;
        for (int j = 0; j < count; ++j) {
            result[j] = REVERSE[src[i] & 0xFF];
            ++i;
        }
        return result;
    }

    public static void reverseBitOrder(byte[] dest, int destPos, byte[] src, int srcPos, int count) {
        Objects.requireNonNull(dest, "Null dest");
        Objects.requireNonNull(src, "Null src");
        JArrays.rangeCheck(src.length, srcPos, dest.length, destPos, count);
        int i = srcPos;
        int j = destPos;
        int toIndex = srcPos + count;
        while (i < toIndex) {
            dest[j] = REVERSE[src[i] & 0xFF];
            ++i;
            ++j;
        }
    }

    public static byte reverseBitOrder(byte value) {
        return REVERSE[value & 0xFF];
    }
}

