/*
 * Decompiled with CFR 0.152.
 */
package net.algart.arrays;

import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import net.algart.arrays.AbstractArrayProcessorWithContextSwitching;
import net.algart.arrays.ArrayContext;
import net.algart.arrays.Arrays;
import net.algart.arrays.ArraysTileMatrixImpl;
import net.algart.arrays.BitArray;
import net.algart.arrays.Matrix;
import net.algart.arrays.MemoryModel;
import net.algart.arrays.PArray;
import net.algart.arrays.PFixedArray;
import net.algart.arrays.SimpleMemoryModel;
import net.algart.arrays.SizeMismatchException;
import net.algart.arrays.ThreadPoolFactory;
import net.algart.arrays.UpdatableBitArray;
import net.algart.arrays.UpdatablePArray;
import net.algart.math.Range;
import net.algart.math.functions.ConstantFunc;

public class GeneralizedBitProcessing
extends AbstractArrayProcessorWithContextSwitching {
    private final ThreadPoolFactory threadPoolFactory;
    private final int numberOfTasks;
    private final SliceOperation sliceOperation;
    private final RoundingMode roundingMode;
    private final boolean inPlaceProcessingAllowed;

    private GeneralizedBitProcessing(ArrayContext context, SliceOperation sliceOperation, RoundingMode roundingMode, int numberOfTasks) {
        super(context);
        Objects.requireNonNull(sliceOperation, "Null sliceOperation argument");
        Objects.requireNonNull(roundingMode, "Null roundingMode argument");
        this.threadPoolFactory = Arrays.getThreadPoolFactory(context);
        this.numberOfTasks = numberOfTasks > 0 ? numberOfTasks : Math.max(1, this.threadPoolFactory.recommendedNumberOfTasks());
        this.sliceOperation = sliceOperation;
        this.roundingMode = roundingMode;
        this.inPlaceProcessingAllowed = sliceOperation.isInPlaceProcessingAllowed();
    }

    public static GeneralizedBitProcessing getInstance(ArrayContext context, SliceOperation sliceOperation, RoundingMode roundingMode) {
        return new GeneralizedBitProcessing(context, sliceOperation, roundingMode, 0);
    }

    public static GeneralizedBitProcessing getSingleThreadInstance(ArrayContext context, SliceOperation sliceOperation, RoundingMode roundingMode) {
        return new GeneralizedBitProcessing(context, sliceOperation, roundingMode, 1);
    }

    @Override
    public GeneralizedBitProcessing context(ArrayContext newContext) {
        return newContext == this.context() ? this : new GeneralizedBitProcessing(newContext, this.sliceOperation, this.roundingMode, this.numberOfTasks);
    }

    public SliceOperation sliceOperation() {
        return this.sliceOperation;
    }

    public RoundingMode roundingMode() {
        return this.roundingMode;
    }

    public int numberOfTasks() {
        return this.numberOfTasks;
    }

    public void process(UpdatablePArray dest, PArray src, Range range, long numberOfSlices) {
        Objects.requireNonNull(dest, "Null dest argument");
        Objects.requireNonNull(src, "Null src argument");
        Objects.requireNonNull(range, "Null range argument");
        if (src.elementType() != dest.elementType()) {
            throw new IllegalArgumentException("Different element types of src and dest arrays (" + src.elementType().getCanonicalName() + " and " + dest.elementType().getCanonicalName() + ")");
        }
        if (dest.length() != src.length()) {
            throw new SizeMismatchException("Different lengths of src and dest arrays (" + src.length() + " and " + dest.length() + ")");
        }
        if (numberOfSlices <= 0L) {
            throw new IllegalArgumentException("numberOfSlices must be positive");
        }
        if (src instanceof PFixedArray) {
            numberOfSlices = Math.min(numberOfSlices, (long)(range.size() + 1.0));
        }
        if (numberOfSlices >= 2L && src instanceof BitArray) {
            if (this.inPlaceProcessingAllowed) {
                Arrays.copy(this.contextPart(0.0, 0.05), dest, src);
                this.sliceOperation.processBits(this.contextPart(0.05, 1.0), (UpdatableBitArray)dest, (BitArray)((Object)dest), 1L, 0, 1);
            } else {
                this.sliceOperation.processBits(this.context(), (UpdatableBitArray)dest, (BitArray)src, 1L, 0, 1);
            }
            return;
        }
        long numberOfRanges = numberOfSlices - 1L;
        ArrayContext acFill = numberOfRanges == 0L ? this.context() : this.contextPart(0.0, 0.1 / (double)numberOfRanges);
        ConstantFunc filler = switch (this.roundingMode.ordinal()) {
            case 0 -> ConstantFunc.getInstance(range.min());
            case 1 -> ConstantFunc.getInstance(range.max());
            default -> throw new AssertionError();
        };
        Arrays.copy(acFill, dest, Arrays.asIndexFuncArray(filler, dest.type(), dest.length()));
        if (numberOfRanges == 0L) {
            return;
        }
        assert (numberOfSlices >= 2L);
        Runnable[] tasks = this.createTasks(this.contextPart(0.1 / (double)numberOfRanges, 1.0), dest, src, range, numberOfRanges);
        this.threadPoolFactory.performTasks(tasks);
    }

    private Runnable[] createTasks(ArrayContext context, UpdatablePArray dest, PArray src, Range range, long numberOfRanges) {
        long len = src.length();
        assert (dest.length() == len);
        assert (dest.elementType() == src.elementType());
        assert (numberOfRanges >= 1L) : "1 slice must be processed by trivial filling out of this method";
        int nt = (int)Math.min((long)this.numberOfTasks, numberOfRanges);
        UpdatableBitArray[] srcBits = this.allocateMemory(nt, src);
        UpdatableBitArray[] destBits = this.inPlaceProcessingAllowed ? srcBits : this.allocateMemory(nt, src);
        Runnable[] tasks = new Runnable[nt];
        AtomicLong readyLayers = new AtomicLong(0L);
        AtomicBoolean interruptionRequest = new AtomicBoolean(false);
        if (context != null && nt > 1) {
            context = context.singleThreadVersion();
        }
        ReentrantLock lock = new ReentrantLock();
        Condition condition = lock.newCondition();
        for (int threadIndex = 0; threadIndex < tasks.length; ++threadIndex) {
            SliceProcessingTask task = new SliceProcessingTask(context, threadIndex, nt, lock, condition);
            task.setProcessedData(dest, src, destBits, srcBits);
            task.setRanges(range, numberOfRanges);
            task.setSynchronizationVariables(readyLayers, interruptionRequest);
            tasks[threadIndex] = task;
        }
        return tasks;
    }

    private UpdatableBitArray[] allocateMemory(int numberOfTasks, PArray src) {
        long nba = this.sliceOperation.isInPlaceProcessingAllowed() ? (long)numberOfTasks : 2L * (long)numberOfTasks;
        long arrayLength = src.length();
        MemoryModel mm = Arrays.sizeOf(Boolean.TYPE, arrayLength) < Arrays.SystemSettings.maxTempJavaMemory() / nba ? SimpleMemoryModel.getInstance() : this.memoryModel();
        UpdatableBitArray[] result = new UpdatableBitArray[numberOfTasks];
        Matrix<PArray> srcBaseMatrix = !Arrays.isTiled(src) ? null : ((ArraysTileMatrixImpl.TileMatrixArray)((Object)src)).baseMatrix().cast(PArray.class);
        for (int k = 0; k < result.length; ++k) {
            result[k] = mm.newUnresizableBitArray(arrayLength);
            if (srcBaseMatrix == null) continue;
            result[k] = srcBaseMatrix.matrix(result[k]).tile(Arrays.tileDimensions(src)).array();
        }
        return result;
    }

    public static interface SliceOperation {
        public void processBits(ArrayContext var1, UpdatableBitArray var2, BitArray var3, long var4, int var6, int var7);

        public boolean isInPlaceProcessingAllowed();
    }

    public static enum RoundingMode {
        ROUND_DOWN,
        ROUND_UP;

    }

    private class SliceProcessingTask
    implements Runnable {
        private final ArrayContext context;
        private final int threadIndex;
        private final int numberOfThreads;
        private final Lock lock;
        private final Condition condition;
        private Range range;
        private long numberOfRanges;
        private AtomicLong readyLayers;
        private AtomicBoolean interruptionRequest;
        private UpdatablePArray dest;
        private PArray src;
        private UpdatableBitArray[] srcBits;
        private UpdatableBitArray[] destBits;

        private SliceProcessingTask(ArrayContext context, int threadIndex, int numberOfThreads, Lock lock, Condition condition) {
            this.context = context;
            this.threadIndex = threadIndex;
            this.numberOfThreads = numberOfThreads;
            this.lock = lock;
            this.condition = condition;
        }

        public void setRanges(Range range, long numberOfRanges) {
            this.range = range;
            this.numberOfRanges = numberOfRanges;
        }

        private void setProcessedData(UpdatablePArray dest, PArray src, UpdatableBitArray[] destBits, UpdatableBitArray[] srcBits) {
            this.dest = dest;
            this.src = src;
            this.destBits = destBits;
            this.srcBits = srcBits;
        }

        private void setSynchronizationVariables(AtomicLong readyLayers, AtomicBoolean interruptionRequest) {
            this.readyLayers = readyLayers;
            this.interruptionRequest = interruptionRequest;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public void run() {
            if (this.lock == null) throw new AssertionError((Object)(String.valueOf(this) + " is not initialized correctly"));
            if (this.range == null) throw new AssertionError((Object)(String.valueOf(this) + " is not initialized correctly"));
            if (this.dest == null) throw new AssertionError((Object)(String.valueOf(this) + " is not initialized correctly"));
            if (this.src == null) {
                throw new AssertionError((Object)(String.valueOf(this) + " is not initialized correctly"));
            }
            long sliceIndex = this.threadIndex + 1;
            while (sliceIndex <= this.numberOfRanges) {
                ArrayContext acFromBit;
                ArrayContext acOp;
                ArrayContext acToBit;
                double value = switch (GeneralizedBitProcessing.this.roundingMode.ordinal()) {
                    case 0 -> {
                        if (sliceIndex == this.numberOfRanges) {
                            yield this.range.max();
                        }
                        yield this.range.min() + this.range.size() * (double)sliceIndex / (double)this.numberOfRanges;
                    }
                    case 1 -> {
                        if (sliceIndex == this.numberOfRanges) {
                            yield this.range.min();
                        }
                        yield this.range.max() - this.range.size() * (double)sliceIndex / (double)this.numberOfRanges;
                    }
                    default -> throw new AssertionError();
                };
                if (this.context != null) {
                    ArrayContext ac = this.context.part(sliceIndex - 1L, sliceIndex, this.numberOfRanges);
                    acToBit = this.numberOfThreads == 1 ? ac.part(0.0, 0.05) : this.context.noProgressVersion();
                    acOp = this.numberOfThreads == 1 ? ac.part(0.05, 0.95) : this.context.noProgressVersion();
                    acFromBit = ac.part(0.95, 1.0);
                } else {
                    acOp = acFromBit = ArrayContext.DEFAULT_SINGLE_THREAD;
                    acToBit = acFromBit;
                }
                if (this.interruptionRequest.get()) {
                    return;
                }
                switch (GeneralizedBitProcessing.this.roundingMode.ordinal()) {
                    case 0: {
                        Arrays.packBitsGreaterOrEqual(acToBit, this.srcBits[this.threadIndex], this.src, value);
                        break;
                    }
                    case 1: {
                        Arrays.packBitsGreater(acToBit, this.srcBits[this.threadIndex], this.src, value);
                        break;
                    }
                }
                GeneralizedBitProcessing.this.sliceOperation.processBits(acOp, this.destBits[this.threadIndex], this.srcBits[this.threadIndex], GeneralizedBitProcessing.this.roundingMode == RoundingMode.ROUND_DOWN ? sliceIndex : this.numberOfRanges - sliceIndex, this.threadIndex, this.numberOfThreads);
                this.lock.lock();
                try {
                    if (this.interruptionRequest.get()) {
                        return;
                    }
                    while (this.readyLayers.get() < sliceIndex - 1L) {
                        try {
                            this.condition.await(100L, TimeUnit.MILLISECONDS);
                        }
                        catch (InterruptedException e) {
                            this.interruptionRequest.set(true);
                            this.lock.unlock();
                            return;
                        }
                        if (this.context == null) continue;
                        this.context.checkInterruption();
                    }
                    switch (GeneralizedBitProcessing.this.roundingMode.ordinal()) {
                        case 0: {
                            Arrays.unpackUnitBits(acFromBit, this.dest, this.destBits[this.threadIndex], value);
                            break;
                        }
                        case 1: {
                            Arrays.unpackZeroBits(acFromBit, this.dest, this.destBits[this.threadIndex], value);
                            break;
                        }
                    }
                    long ready = this.readyLayers.incrementAndGet();
                    if (ready != sliceIndex) {
                        throw new AssertionError((Object)("Invalid synchronization in " + String.valueOf(GeneralizedBitProcessing.class)));
                    }
                    this.condition.signalAll();
                }
                finally {
                    this.lock.unlock();
                }
                sliceIndex += (long)this.numberOfThreads;
            }
        }
    }
}

